import React, { useState, useCallback, useEffect } from 'react';
import { useParams } from 'react-router';
import {
  AllowlistEntry,
  ListEntriesByOrgRequest,
  DeleteAllowlistEntryRequest,
  UpdateAllowlistEntryRequest,
} from '~core/clients/twitch-e2-ingest-http/twitch.fulton.example';
import { TwitchE2IngestHTTPClient } from '~core/vienna';
import { toast } from 'react-toastify';
import { OrganizationE2Presentation } from '~pages/organization-e2/organization-e2-presentation';
import { E2ModalSchema } from '~features/allowlist-e2-modal/allowlist-e2-modal';

interface RouteProps {
  organizationId: string;
}

export const OrganizationE2Page: React.FC = () => {
  const [entries, setEntries] = useState<AllowlistEntry[]>([]);
  const { organizationId } = useParams<RouteProps>();

  async function getEntries() {
    const response = await TwitchE2IngestHTTPClient.listEntriesByOrg(
      new ListEntriesByOrgRequest({ organizationId: organizationId }),
    );
    if (response) {
      setEntries(response.entries);
    }
  }

  const getEntriesCallback = useCallback(getEntries, [organizationId]);

  useEffect(() => {
    getEntriesCallback();
  }, [getEntriesCallback, organizationId]);

  async function updateEntry(request: E2ModalSchema) {
    try {
      await TwitchE2IngestHTTPClient.updateAllowlistEntry(
        new UpdateAllowlistEntryRequest({ ...request }),
      );

      toast('Successfully updated entry', { type: 'success' });
      getEntries();
    } catch (err) {
      toast(`Failed to update entry: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function deleteEntry(entry: AllowlistEntry) {
    try {
      await TwitchE2IngestHTTPClient.deleteAllowlistEntry(
        new DeleteAllowlistEntryRequest({
          gameId: entry.gameId,
          clientId: entry.clientId,
        }),
      );

      toast('Successfully deleted entry', { type: 'success' });
      getEntries();
    } catch (err) {
      toast(`Failed to delete entry: ${(err as Error).message}`, { type: 'error' });
    }
  }

  return (
    <>
      <OrganizationE2Presentation
        entries={entries}
        onEditEntry={updateEntry}
        onDeleteEntry={deleteEntry}
      />
    </>
  );
};
