import React, { useState, useCallback, useEffect } from 'react';
import { OrganizationGamesPresentation } from '~pages/organization-games/organization-games-presentation';
import { Game, Id, OffboardResourceRequest } from '~core/clients/rbac/code.justin.tv.devrel';
import { useParams } from 'react-router';
import { RBACClient, RBACAdminClient } from '~core/vienna';
import { useModal } from 'tachyon-modal';
import { ActionHistoryModal } from '~features/action-history-modal';
import { ActionHistoryEntityType } from '~features/action-history/models/entity-type';
import { toast } from 'react-toastify';

interface RouteProps {
  organizationId: string;
}

export const OrganizationGamesPage: React.FC = () => {
  const [games, setGames] = useState<Game[]>([]);
  const [selectedGame, setSelectedGame] = useState<string | undefined>(undefined);

  const { openModal, closeModal, id, active } = useModal();

  const { organizationId } = useParams<RouteProps>();

  async function getGames() {
    const gamesResponse = await RBACClient.getGamesByCompany(new Id({ id: organizationId }));
    if (gamesResponse) {
      setGames(gamesResponse.games);
    }
  }

  const getGamesCallback = useCallback(getGames, [organizationId]);

  useEffect(() => {
    getGamesCallback();
  }, [getGamesCallback, organizationId]);

  function onActivityModal(gameId: string) {
    setSelectedGame(gameId);
    openModal();
  }

  async function removeGame(gameId: number) {
    try {
      await RBACAdminClient.offboardResource(
        new OffboardResourceRequest({
          resourceType: 'game',
          resourceId: gameId.toString(),
          companyId: organizationId,
        }),
      );

      toast('Successfully offboarded game', { type: 'success' });
      getGames();
    } catch (err) {
      toast(`Failed to remove Game: ${(err as Error).message}`, { type: 'error' });
    }
  }

  return (
    <>
      <OrganizationGamesPresentation
        games={games}
        onActivityModal={onActivityModal}
        onRemoveGame={removeGame}
      />
      <ActionHistoryModal
        companyId={organizationId}
        active={active}
        id={id}
        closeModal={closeModal}
        entityId={selectedGame}
        entityType={ActionHistoryEntityType.Game}
      />
    </>
  );
};
