import React, { useCallback, useState, useEffect } from 'react';
import { useParams } from 'react-router';
import { toast } from 'react-toastify';
import { RBACClient } from '~core/vienna';
import {
  Id,
  CompanyApplication,
  UpdateCompanyApplicationRequest,
  OnboardCompanyRequest,
  DeleteCompanyApplicationRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { Layout, Display, JustifyContent, AlignItems, LoadingSpinner } from 'twitch-core-ui';
import { OrganizationApplicationStatus } from '~features/organization-application/models/organization-application-status';
import { ApplicationPending } from '~pages/organization-review-detail/components/application-pending/application-pending';
import { OrganizationInformation } from '~pages/organization-detail/models/organization-information';
import { ApplicationProcessed } from '~pages/organization-review-detail/components/application-processed';

export interface RouteProps {
  applicationId: string;
}

export const OrganizationReviewDetailPage: React.FC = () => {
  const { applicationId } = useParams<RouteProps>();
  const [application, setApplication] = useState<CompanyApplication>();

  async function loadApplication() {
    try {
      const result = await RBACClient.getCompanyApplication(new Id({ id: applicationId }));
      setApplication(result);
    } catch {
      toast('Error loading application', { type: 'error' });
    }
  }

  const loadApplicationCallback = useCallback(loadApplication, [applicationId]);

  useEffect(() => {
    loadApplicationCallback();
  }, [applicationId, loadApplicationCallback]);

  async function updateOrganizationApplication(values: OrganizationInformation) {
    try {
      const request = UpdateCompanyApplicationRequest.fromJSON({
        ...application?.toJSON(),
        id: applicationId,
        company_name: values.companyName,
        company_website: values.url,
        company_type: values.type,
        games: application?.games.map((g) => g.id.toString()),
      });
      const result = await RBACClient.updateCompanyApplication(request);
      setApplication(result);
      toast('Updated the application!', { type: 'success' });
    } catch (err) {
      toast(`Updating the Application failed: ${(err as Error).message}`, {
        type: 'error',
      });
    }
  }

  async function approveApplication() {
    try {
      await RBACClient.onboardCompany(
        new OnboardCompanyRequest({
          id: applicationId,
        }),
      );
      toast('Accepted Application!', { type: 'success' });
      loadApplication();
    } catch (err) {
      toast(`Error accepting application: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function rejectApplication() {
    try {
      await RBACClient.deleteCompanyApplication(
        new DeleteCompanyApplicationRequest({
          id: applicationId,
        }),
      );
      toast('Rejected Application!', { type: 'success' });
      loadApplication();
    } catch (err) {
      toast(`Error rejecting application; ${(err as Error).message}`, { type: 'error' });
    }
  }

  if (!application) {
    return (
      <Layout
        fullWidth
        fullHeight
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Center}
      >
        <LoadingSpinner />
      </Layout>
    );
  }

  if (application.status === OrganizationApplicationStatus.Pending) {
    return (
      <ApplicationPending
        application={application}
        onUpdateApplication={updateOrganizationApplication}
        onApproveApplication={approveApplication}
        onRejectApplication={rejectApplication}
      />
    );
  } else if (
    application.status === OrganizationApplicationStatus.Approved ||
    application.status === OrganizationApplicationStatus.Rejected
  ) {
    return <ApplicationProcessed application={application} />;
  }
  return null;
};
