import React, { useState, useCallback, useEffect } from 'react';
import { usePagination } from '~features/hooks/use-pagination';
import { RBACClient } from '~core/vienna';
import {
  ListCompanyApplicationsRequest,
  CompanyApplication,
  SearchCompanyApplicationsRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { toast } from 'react-toastify';
import { DebounceTimer } from '~core/utils/debounce-timer';
import { OrganizationApplicationStatus } from '~features/organization-application/models/organization-application-status';
import { EntityList } from '~features/entity-list/entity-list';
import { OrganizationApplicationListItem } from '~features/organization-application/components/organization-application-list-item';
import { Layout, Display, AlignItems, CoreText, FontWeight, Color, Select } from 'twitch-core-ui';

const ORGANIZATION_LIMIT = 25;
const searchDebouncer = new DebounceTimer(200);

export const OrganizationReviewListPage: React.FC = () => {
  const { page, nextPage, previousPage, goToPage } = usePagination();
  const [searchTerm, setSearchTerm] = useState('');
  const [applications, setApplications] = useState<CompanyApplication[]>([]);
  const [total, setTotal] = useState(100);
  const [loading, setloading] = useState(false);
  const [applicationStatus, setApplicationStatus] = useState(OrganizationApplicationStatus.Pending);

  async function getApplications() {
    setloading(true);

    try {
      const result = await RBACClient.listCompanyApplications(
        new ListCompanyApplicationsRequest({
          limit: ORGANIZATION_LIMIT,
          offset: ORGANIZATION_LIMIT * (page - 1),
          status: applicationStatus,
        }),
      );
      setTotal(Math.ceil(result.Total / ORGANIZATION_LIMIT));
      setApplications(result.companyApplications);
    } catch (err) {
      toast('Failed loading Applications', { type: 'error' });
    } finally {
      setloading(false);
    }
  }
  const getApplicationsCallback = useCallback(getApplications, [page, applicationStatus]);

  async function searchCompanyApplication() {
    if (!searchTerm.length) {
      await getApplicationsCallback();
      return;
    }

    try {
      setloading(true);
      const result = await RBACClient.searchCompanyApplications(
        new SearchCompanyApplicationsRequest({
          query: searchTerm,
          limit: ORGANIZATION_LIMIT,
        }),
      );

      setTotal(Math.ceil(result.Total / ORGANIZATION_LIMIT));
      setApplications(result.companyApplications);
    } catch (err) {
      toast('Failed to search for applications', { type: 'error' });
    } finally {
      setloading(false);
    }
  }

  const searchCompanyApplicationCallback = useCallback(searchCompanyApplication, [
    searchTerm,
    getApplicationsCallback,
  ]);

  useEffect(() => {
    getApplicationsCallback();
  }, [page, getApplicationsCallback]);

  // useEffect(() => {
  //   goToPage(1);
  // }, [applicationStatus]);

  useEffect(() => {
    searchDebouncer.invoke(searchCompanyApplicationCallback);
  }, [searchCompanyApplicationCallback, searchTerm]);

  return (
    <EntityList<CompanyApplication>
      pagination={{
        currentPage: page,
        onNextPage: nextPage,
        onPreviousPage: previousPage,
        onGoToPage: goToPage,
        totalPages: total,
      }}
      columns={['Name', 'Games Pending', 'Status', 'Created At']}
      items={applications}
      zeroStateMessage="No Applications Available"
      rowType={OrganizationApplicationListItem}
      rowProps={{}}
      loading={loading}
      header={{
        searchTerm: searchTerm,
        onSearchUpdate: setSearchTerm,
        searchPlaceholder: 'Search Applications',
        additionalFields: [
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout margin={{ right: 2 }}>
              <CoreText fontWeight={FontWeight.SemiBold} color={Color.Alt2}>
                Status
              </CoreText>
            </Layout>
            <Select
              value={applicationStatus.toString()}
              onChange={(ev) => setApplicationStatus(parseInt(ev.currentTarget.value, 10))}
            >
              <option value="1">Pending</option>
              <option value="2">Rejected</option>
              <option value="3">Approved</option>
            </Select>
          </Layout>,
        ],
      }}
    />
  );
};
