package companies

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/developerapplications"
	"code.justin.tv/devrel/devsite-rbac/backend/gameapplications"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/Masterminds/squirrel"
	"github.com/cactus/go-statsd-client/statsd"
)

const Table = "companies"

//go:generate counterfeiter . CBackender
//go:generate errxer --timings CBackender
type CBackender interface {
	FindCompanies(ctx context.Context, params *rbacrpc.ListCompaniesRequest) (*rbacrpc.ListCompaniesResponse, error)
	SearchCompanies(ctx context.Context, query string, limit uint64) (*rbacrpc.SearchCompaniesResponse, error)
	SelectCompanies(ctx context.Context, ids []string) (map[string]*rbacrpc.Company, error)
	SelectCompany(ctx context.Context, id string) (*rbacrpc.Company, error)
	SelectCompanyByCompanyName(ctx context.Context, name string) (*rbacrpc.Company, error)

	InsertCompany(ctx context.Context, params *CreateRequest) (*rbacrpc.Company, error)
	UpdateCompany(ctx context.Context, params *rbacrpc.UpdateCompanyRequest) (*rbacrpc.Company, error)
	DeleteCompany(ctx context.Context, params *rbacrpc.DeleteCompanyRequest) error

	// CompanyExistsWithIdentifier finds by identifier, which should be generated from the name with common.Identifier,
	// and returns true if the company already exists; false if does not exist or an error was returned.
	CompanyExistsWithIdentifier(ctx context.Context, identifier string) (bool, error)
}

// Company represents a row in the companies table.
// Our version of Company is the third iteration for company data.
// A long time ago, there was a legacy company. CurseForge became
// the owner of company data and imported companies have Legacy set
// to true and have a CurseCompanyId. Companies created in CurseForge
// have Legacy set to false and have a CurseCompanyId. Enter RBAC,
// the current owner of company data. Companies created by RBAC have
// Legacy set to false and CurseCompanyId set to -1.
type Company struct {
	ID                string `db:"id"`
	CompanyName       string `db:"company_name"`
	URL               string `db:"url"`
	Type              int32  `db:"type"`
	VhsContractSigned bool   `db:"vhs_contract_signed"`
	CampaignsEnabled  bool   `db:"campaigns_enabled"`
	Identifier        string `db:"identifier"`
	// Whether or not the company was migrated into CurseForge.
	Legacy bool `db:"legacy"`
	// Set if the company is legacy or was created by CurseForge.
	CurseCompanyID sql.NullInt64 `db:"curse_company_id"`
	CreatedAt      string        `db:"created_at"`

	XXX_Total int32 `db:"_total"`
}

var Columns = dbx.FieldsFrom(&Company{}).Exclude("_total")

type backend struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter) *CBackenderErrx {
	return &CBackenderErrx{
		CBackender: &backend{db: db},
		TimingFunc: common.TimingStats(stats),
	}
}

func (b *backend) FindCompanies(ctx context.Context, params *rbacrpc.ListCompaniesRequest) (*rbacrpc.ListCompaniesResponse, error) {
	cols := Columns.Add(common.CountOverAs("_total"))

	q := common.PSQL.Select(cols...).From(Table)

	if params.HasGamesPending {
		gameAppSQL := common.PSQL.Select("company_id").Distinct().From(gameapplications.Table)
		q = common.PSQL.Select(cols...).FromSelect(gameAppSQL, "G").
			Join("companies ON G.company_id = companies.id")
	}

	if params.HasDevelopersPending {
		devAppSQL := common.PSQL.Select("company_id").Distinct().From(developerapplications.Table)
		q = common.PSQL.Select(cols...).FromSelect(devAppSQL, "D").
			Join("companies ON D.company_id = companies.id")
	}

	q = common.Paginate(q, params.Limit, params.Offset)
	q = q.OrderBy("company_name")

	var companies []Company
	err := b.db.LoadAll(ctx, &companies, q)
	if err != nil {
		return nil, err
	}

	return &rbacrpc.ListCompaniesResponse{
		Companies: ListToRPC(companies),
		XTotal:    common.FirstRowInt32DBField(companies, "_total"),
	}, nil
}

func (b *backend) SearchCompanies(ctx context.Context, query string, limit uint64) (*rbacrpc.SearchCompaniesResponse, error) {
	cols := Columns.Add(common.CountOverAs("_total"))

	q := common.PSQL.Select(cols...).From(Table).
		// query is added as an arg twice so it's used in the OrderBy statement
		Where("similarity(company_name, (?)) > .1", query, query).
		OrderBy("similarity(company_name, (?)) DESC").
		Limit(limit)

	var companies []Company
	err := b.db.LoadAll(ctx, &companies, q)
	if err != nil {
		return nil, err
	}

	return &rbacrpc.SearchCompaniesResponse{
		Companies: ListToRPC(companies),
		XTotal:    common.FirstRowInt32DBField(companies, "_total"),
	}, nil
}

func (b *backend) SelectCompanies(ctx context.Context, ids []string) (map[string]*rbacrpc.Company, error) {
	q := common.PSQL.Select(Columns...).From(Table).Where(squirrel.Eq{"id": ids})

	var companies []Company
	err := b.db.LoadAll(ctx, &companies, q)
	if err != nil {
		return nil, err
	}

	companiesByID := make(map[string]*rbacrpc.Company)
	for _, company := range companies {
		companiesByID[company.ID] = company.ToRPC()
	}
	return companiesByID, nil
}

func (b *backend) SelectCompany(ctx context.Context, id string) (*rbacrpc.Company, error) {
	q := common.PSQL.Select(Columns...).From(Table).Where("id = ?", id)
	var c Company
	err := b.db.LoadOne(ctx, &c, q)
	return c.ToRPC(), err
}

func (b *backend) SelectCompanyByCompanyName(ctx context.Context, companyName string) (*rbacrpc.Company, error) {
	q := common.PSQL.Select(Columns...).From(Table).Where("company_name = ?", companyName)
	var c Company
	err := b.db.LoadOne(ctx, &c, q)
	return c.ToRPC(), err
}

type CreateRequest struct {
	CompanyName       string
	Url               string
	Type              int32
	VhsContractSigned bool
	CampaignsEnabled  bool
	Identifier        string
	ApplicationId     string
	CurseCompanyId    int32
}

func (b *backend) InsertCompany(ctx context.Context, params *CreateRequest) (*rbacrpc.Company, error) {
	// We reuse applicationId as the companyId so calls to pushy can use a consistent identifier for the EVSKey. A
	// consistent identifier is required for the EVS (email validation service) to be used as the key tied to the
	// email being verified. If the key were to lose consistency then a verified email will no longer be recognized
	// as verified.
	id := params.ApplicationId
	if id == "" {
		id = common.NewUUID()
	}

	company := &Company{
		ID:                id,
		CompanyName:       params.CompanyName,
		URL:               params.Url,
		Type:              params.Type,
		VhsContractSigned: params.VhsContractSigned,
		CampaignsEnabled:  params.CampaignsEnabled,
		Identifier:        params.Identifier,
		CurseCompanyID:    common.NewSQLNullInt64(int64(params.CurseCompanyId)),
		CreatedAt:         time.Now().Format(time.RFC3339),
	}
	err := b.db.InsertOne(ctx, Table, company, dbx.Exclude("_total"))
	return company.ToRPC(), err
}

func (b *backend) UpdateCompany(ctx context.Context, params *rbacrpc.UpdateCompanyRequest) (*rbacrpc.Company, error) {
	c := Company{
		ID:                params.Id,
		CompanyName:       params.CompanyName,
		URL:               params.Url,
		Type:              params.Type,
		VhsContractSigned: params.VhsContractSigned,
		CampaignsEnabled:  params.CampaignsEnabled,
		Identifier:        params.Identifier,
		CurseCompanyID:    common.NewSQLNullInt64(int64(params.CurseCompanyId)),
	}
	err := b.db.UpdateOne(ctx, Table, &c, dbx.FindBy("id"), dbx.Exclude("created_at", "legacy", "_total"))
	return c.ToRPC(), err
}

func (b *backend) DeleteCompany(ctx context.Context, params *rbacrpc.DeleteCompanyRequest) error {
	return b.db.DeleteOne(ctx, Table, dbx.Values{"id": params.Id})
}

func (b *backend) CompanyExistsWithIdentifier(ctx context.Context, identifier string) (bool, error) {
	var c Company
	err := b.db.LoadOne(ctx, &c,
		common.PSQL.Select("id", "identifier").From(Table).Where("identifier = ?", identifier))
	if errorutil.IsErrNoRows(err) {
		return false, nil // does not exist
	}
	if err != nil {
		return false, err // some other internal error
	}
	return true, nil // already exists
}

//
// Converters
//

func (c Company) ToRPC() *rbacrpc.Company {
	rpcCompany := &rbacrpc.Company{
		Id:                c.ID,
		CompanyName:       c.CompanyName,
		Url:               c.URL,
		Type:              c.Type,
		VhsContractSigned: c.VhsContractSigned,
		CampaignsEnabled:  c.CampaignsEnabled,
		Identifier:        c.Identifier,
		CreatedAt:         c.CreatedAt,
		Legacy:            c.Legacy,
	}

	if c.CurseCompanyID.Valid {
		rpcCompany.CurseCompanyId = int32(c.CurseCompanyID.Int64)
	}

	return rpcCompany
}

func ListToRPC(list []Company) []*rbacrpc.Company {
	companies := make([]*rbacrpc.Company, len(list))
	for i, company := range list {
		companies[i] = company.ToRPC()
	}
	return companies
}
