package companyapplications

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/models"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/cactus/go-statsd-client/statsd"
)

const Table = "company_applications"

//go:generate counterfeiter . CompanyApplications
//go:generate errxer --timings CompanyApplications
type CompanyApplications interface {
	ListCompanyApplications(ctx context.Context, params *rbacrpc.ListCompanyApplicationsRequest) ([]CompanyApplication, int32, error)
	SearchCompanyApplications(ctx context.Context, query string, limit uint64) ([]CompanyApplication, int32, error)
	GetCompanyApplication(ctx context.Context, id string) (CompanyApplication, error)

	InsertCompanyApplication(ctx context.Context, params *rbacrpc.CreateCompanyApplicationRequest) (*rbacrpc.CompanyApplication, error)
	UpdateCompanyApplication(ctx context.Context, params *rbacrpc.UpdateCompanyApplicationRequest) (*rbacrpc.CompanyApplication, error)
	UpdateCompanyApplicationStatus(ctx context.Context, id string, status int64) error
	DeleteCompanyApplicationsByAccount(ctx context.Context, twitchID string) error
}

type CompanyApplication struct {
	Id               string         `db:"id"` // will become the company id
	TwitchId         string         `db:"twitch_id"`
	Status           int32          `db:"status"`
	CompanyName      string         `db:"company_name"`
	CompanyWebsite   string         `db:"company_website"`
	CompanyType      int32          `db:"company_type"`
	Games            string         `db:"games"` // comma sepparated list of game ids
	Industry         string         `db:"industry"`
	CompanySize      string         `db:"company_size"`
	City             string         `db:"city"`
	State            string         `db:"state"`
	Country          string         `db:"country"`
	ContactFirstName string         `db:"contact_first_name"`
	ContactLastName  string         `db:"contact_last_name"`
	ContactTitle     string         `db:"contact_title"`
	ContactEmail     string         `db:"contact_email"`
	ProductInterest  sql.NullString `db:"product_interest"`
	JoinReason       sql.NullString `db:"join_reason"`
	CreatedAt        sql.NullString `db:"created_at"`
	UpdatedAt        sql.NullString `db:"updated_at"`

	XXX_Total int32 `db:"_total"`
}

type dbxCompApp struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter) CompanyApplications {
	return &SecureBackend{
		CompanyApplications: &CompanyApplicationsErrx{
			CompanyApplications: &dbxCompApp{db: db},
			TimingFunc:          common.TimingStats(stats),
		},
	}
}

var Columns = dbx.FieldsFrom(CompanyApplication{}).Exclude("_total")

func (b *dbxCompApp) ListCompanyApplications(ctx context.Context, params *rbacrpc.ListCompanyApplicationsRequest) ([]CompanyApplication, int32, error) {
	cols := Columns.Add(common.CountOverAs("_total"))
	q := common.PSQL.Select(cols...).From(Table)
	if params.Status != 0 {
		q = q.Where("status = ?", params.Status)
	}
	if params.TwitchId != "" {
		q = q.Where("twitch_id = ?", params.TwitchId)
	}
	q = common.Paginate(q, params.Limit, params.Offset)
	q = q.OrderBy("created_at DESC")

	capps := []CompanyApplication{}
	err := b.db.LoadAll(ctx, &capps, q)
	return capps, common.FirstRowInt32DBField(capps, "_total"), err
}

func (b *dbxCompApp) SearchCompanyApplications(ctx context.Context, query string, limit uint64) ([]CompanyApplication, int32, error) {
	cols := Columns.Add(common.CountOverAs("_total"))
	q := common.PSQL.Select(cols...).From(Table).
		Where("similarity(company_name, (?)) > .3", query).
		Limit(limit)

	capps := []CompanyApplication{}
	err := b.db.LoadAll(ctx, &capps, q)
	return capps, common.FirstRowInt32DBField(capps, "_total"), err
}

func (b *dbxCompApp) GetCompanyApplication(ctx context.Context, id string) (CompanyApplication, error) {
	q := common.PSQL.Select(Columns...).From(Table).Where("id = ?", id).Limit(1)
	var capp CompanyApplication
	err := b.db.LoadOne(ctx, &capp, q)
	return capp, err
}

func (b *dbxCompApp) InsertCompanyApplication(ctx context.Context, params *rbacrpc.CreateCompanyApplicationRequest) (*rbacrpc.CompanyApplication, error) {
	capp := CompanyApplication{
		Id:     common.NewUUID(),                 // generated CompanyID
		Status: models.CompanyApplicationPending, // pending

		TwitchId:         params.TwitchId,
		CompanyName:      params.CompanyName,
		CompanyWebsite:   params.CompanyWebsite,
		CompanyType:      params.CompanyType,
		Games:            strings.Join(params.Games, ","),
		Industry:         params.Industry,
		CompanySize:      params.CompanySize,
		City:             params.City,
		State:            params.State,
		Country:          params.Country,
		ContactFirstName: params.ContactFirstName,
		ContactLastName:  params.ContactLastName,
		ContactTitle:     params.ContactTitle,
		ContactEmail:     params.ContactEmail,
		ProductInterest:  common.NewSQLNullString(params.ProductInterest),
		JoinReason:       common.NewSQLNullString(params.JoinReason),
		CreatedAt:        common.NewSQLNullString(common.TimeNowStr()),
	}
	err := b.db.InsertOne(ctx, Table, capp,
		dbx.Exclude("updated_at", "_total"))
	return capp.ToRPC(), err
}

func (b *dbxCompApp) UpdateCompanyApplication(ctx context.Context, params *rbacrpc.UpdateCompanyApplicationRequest) (*rbacrpc.CompanyApplication, error) {
	capp := CompanyApplication{
		Id:               params.Id,
		TwitchId:         params.TwitchId,
		Status:           params.Status,
		CompanyName:      params.CompanyName,
		CompanyWebsite:   params.CompanyWebsite,
		CompanyType:      params.CompanyType,
		Games:            strings.Join(params.Games, ","),
		Industry:         params.Industry,
		CompanySize:      params.CompanySize,
		City:             params.City,
		State:            params.State,
		Country:          params.Country,
		ContactFirstName: params.ContactFirstName,
		ContactLastName:  params.ContactLastName,
		ContactTitle:     params.ContactTitle,
		ContactEmail:     params.ContactEmail,
		ProductInterest:  common.NewSQLNullString(params.ProductInterest),
		JoinReason:       common.NewSQLNullString(params.JoinReason),
		UpdatedAt:        common.NewSQLNullString(common.TimeNowStr()),
	}
	err := b.db.UpdateOne(ctx, Table, capp, dbx.FindBy("id"),
		dbx.Exclude("created_at", "_total"))
	return capp.ToRPC(), err
}

func (b *dbxCompApp) UpdateCompanyApplicationStatus(ctx context.Context, id string, status int64) error {
	return b.db.UpdateOne(ctx, Table, dbx.Values{
		"id":         id,
		"status":     status,
		"updated_at": common.NewSQLNullString(common.TimeNowStr()),
	}, dbx.FindBy("id"))
}

func (b *dbxCompApp) DeleteCompanyApplicationsByAccount(ctx context.Context, twitchID string) error {
	da := CompanyApplication{
		TwitchId: twitchID,
	}

	_, err := b.db.NamedExec(ctx, fmt.Sprintf("DELETE FROM %s WHERE twitch_id = :twitch_id", Table), da)

	return err
}

//
// Converters
//

func (a CompanyApplication) ToRPC() *rbacrpc.CompanyApplication {
	return &rbacrpc.CompanyApplication{
		Id:               a.Id,
		TwitchId:         a.TwitchId,
		Status:           a.Status,
		CompanyName:      a.CompanyName,
		CompanyWebsite:   a.CompanyWebsite,
		CompanyType:      a.CompanyType,
		Games:            ToRPCGamesFromStr(a.Games),
		Industry:         a.Industry,
		CompanySize:      a.CompanySize,
		City:             a.City,
		State:            a.State,
		Country:          a.Country,
		ContactFirstName: a.ContactFirstName,
		ContactLastName:  a.ContactLastName,
		ContactTitle:     a.ContactTitle,
		ContactEmail:     a.ContactEmail,

		ProductInterest: a.ProductInterest.String,
		JoinReason:      a.JoinReason.String,
		CreatedAt:       a.CreatedAt.String,
		UpdatedAt:       a.UpdatedAt.String,
	}
}

func ListToRPC(list []CompanyApplication) []*rbacrpc.CompanyApplication {
	apps := make([]*rbacrpc.CompanyApplication, len(list))
	for i, app := range list {
		apps[i] = app.ToRPC()
	}
	return apps
}

func ToRPCGamesFromStr(gamesStr string) []*rbacrpc.CompanyApplication_Game {
	if gamesStr == "" {
		return nil
	}
	return ToRPCGames(strings.Split(gamesStr, ","))
}

func ToRPCGames(gamesSplit []string) []*rbacrpc.CompanyApplication_Game {
	games := make([]*rbacrpc.CompanyApplication_Game, len(gamesSplit))
	for i, game := range gamesSplit {
		games[i] = &rbacrpc.CompanyApplication_Game{
			Id: game,
		}
	}
	return games
}
