package developerapplications

import (
	"context"
	"fmt"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/cactus/go-statsd-client/statsd"
)

const Table = "developer_applications"

//go:generate counterfeiter . DeveloperApplications
//go:generate errxer --timings DeveloperApplications
type DeveloperApplications interface {
	GetDeveloperApplication(ctx context.Context, id string) (DeveloperApplication, error)
	ListDeveloperApplications(ctx context.Context, p ListDeveloperApplicationsParams) ([]DeveloperApplication, error)

	InsertDeveloperApplication(ctx context.Context, devapp *DeveloperApplication) error
	DeleteDeveloperApplication(ctx context.Context, id string) error
	DeleteDeveloperApplicationByCompany(ctx context.Context, companyID string) error
	DeleteDeveloperApplicationByAccount(ctx context.Context, twitchID string) error
}

type DeveloperApplication struct {
	ID        string `db:"id"`
	CompanyID string `db:"company_id"`
	TwitchID  string `db:"twitch_id"`
	FirstName string `db:"first_name"`
	LastName  string `db:"last_name"`
	Title     string `db:"title"`
	Email     string `db:"email"`
	CreatedAt string `db:"created_at"`
}

var Columns = dbx.FieldsFrom(&DeveloperApplication{})

type DBXDeveloperApplication struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter) DeveloperApplications {
	impl := &DBXDeveloperApplication{db: db}
	errxWrap := &DeveloperApplicationsErrx{
		DeveloperApplications: impl,
		TimingFunc:            common.TimingStats(stats),
	}
	return errxWrap
}

func (d *DBXDeveloperApplication) GetDeveloperApplication(ctx context.Context, id string) (DeveloperApplication, error) {
	q := common.PSQL.Select(Columns...).From(Table).
		Where("id = ?", id).
		OrderBy("created_at ASC").Limit(1)
	var devapp DeveloperApplication
	err := d.db.LoadOne(ctx, &devapp, q)
	return devapp, err
}

type ListDeveloperApplicationsParams struct {
	CompanyID string
	Limit     uint64
	Offset    uint64
}

func (d *DBXDeveloperApplication) ListDeveloperApplications(ctx context.Context, p ListDeveloperApplicationsParams) ([]DeveloperApplication, error) {
	q := common.PSQL.Select(Columns...).From(Table)
	if p.CompanyID != "" {
		q = q.Where("company_id = ?", p.CompanyID)
	}
	q = common.Paginate(q, p.Limit, p.Offset)
	q = q.OrderBy("created_at ASC")

	list := []DeveloperApplication{}
	err := d.db.LoadAll(ctx, &list, q)
	return list, err
}

func (d *DBXDeveloperApplication) InsertDeveloperApplication(ctx context.Context, devapp *DeveloperApplication) error {
	devapp.ID = common.NewUUID()
	devapp.CreatedAt = common.TimeNowStr()
	return d.db.InsertOne(ctx, Table, devapp)
}

func (d *DBXDeveloperApplication) DeleteDeveloperApplication(ctx context.Context, id string) error {
	return d.db.DeleteOne(ctx, Table, dbx.Values{"id": id})
}

func (d *DBXDeveloperApplication) DeleteDeveloperApplicationByCompany(ctx context.Context, companyID string) error {
	da := DeveloperApplication{
		CompanyID: companyID,
	}
	_, err := d.db.NamedExec(ctx, fmt.Sprintf("DELETE FROM %s WHERE company_id = :company_id", Table), da)

	return err
}

func (d *DBXDeveloperApplication) DeleteDeveloperApplicationByAccount(ctx context.Context, twitchID string) error {
	da := DeveloperApplication{
		TwitchID: twitchID,
	}

	_, err := d.db.NamedExec(ctx, fmt.Sprintf("DELETE FROM %s WHERE twitch_id = :twitch_id", Table), da)

	return err
}

//
// Converters
//

func (devapp DeveloperApplication) ToRPC() *rbacrpc.DeveloperApplication {
	return &rbacrpc.DeveloperApplication{
		Id:        devapp.ID,
		CompanyId: devapp.CompanyID,
		TwitchId:  devapp.TwitchID,
		FirstName: devapp.FirstName,
		LastName:  devapp.LastName,
		Title:     devapp.Title,
		Email:     devapp.Email,
		CreatedAt: devapp.CreatedAt,
		// EmailVerificationStatus not in DB, needs to be added with s.EVS.GetVerificationStatus if needed
	}
}

func ListToRPC(list []DeveloperApplication) []*rbacrpc.DeveloperApplication {
	devapps := make([]*rbacrpc.DeveloperApplication, len(list))
	for i, devapp := range list {
		devapps[i] = devapp.ToRPC()
	}
	return devapps
}
