package extensionreviewlogs

import (
	"context"

	"github.com/Masterminds/squirrel"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/lib/pq"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

const TableName = "extension_review_logs"

// ExtensionReviewLogs manages the extension_review_logs DB table.
//go:generate errxer --timings ExtensionReviewLogs
//go:generate counterfeiter . ExtensionReviewLogs
type ExtensionReviewLogs interface {
	// Insert a new log row.
	Insert(ctx context.Context, model *rbacrpc.ExtensionReviewLog) error

	// List all logs for an extension. Returns an empty list if none exist.
	ListByExtensionID(ctx context.Context, extensionID string) ([]*rbacrpc.ExtensionReviewLog, error)

	// Get the last row for the same extension version that was set in review. Useful to link when approving/rejecting.
	GetLastInReview(ctx context.Context, extensionID, version string) (*rbacrpc.ExtensionReviewLog, error)

	// Get the last row with author_notes or author_channel, or nil if not found.
	GetLastWithAuthorNotes(ctx context.Context, authorTwitchID string, extensionID string) (*rbacrpc.ExtensionReviewLog, error)
}

type ExtensionReviewLog struct {
	CreatedAt         string         `db:"created_at"`
	State             string         `db:"state"`
	AuthorTwitchId    string         `db:"author_twitch_id"`
	AuthorNotes       string         `db:"author_notes"`
	AuthorChannel     string         `db:"author_channel"`
	ExtensionId       string         `db:"extension_id"`
	ExtensionVersion  string         `db:"extension_version"`
	ReviewerTwitchId  string         `db:"reviewer_twitch_id"`
	ReviewReasonCodes pq.StringArray `db:"review_reason_codes"`
	ReviewReason      string         `db:"review_reason"`
	SalesforceCaseId  string         `db:"salesforce_case_id"`
}

var columns = dbx.FieldsFrom(ExtensionReviewLog{})

type ExtensionReviewLogsDBX struct {
	DBX common.DBXer
}

func New(dbxDB common.DBXer, stats statsd.Statter) ExtensionReviewLogs {
	impl := &ExtensionReviewLogsDBX{DBX: dbxDB}
	wrapper := &ExtensionReviewLogsErrx{
		ExtensionReviewLogs: impl,
		TimingFunc:          common.TimingStats(stats),
	}
	return wrapper
}

func (d *ExtensionReviewLogsDBX) Insert(ctx context.Context, in *rbacrpc.ExtensionReviewLog) error {
	e := ConvertFromRPC(in)
	return d.DBX.InsertOne(ctx, TableName, e)
}

func (d *ExtensionReviewLogsDBX) ListByExtensionID(ctx context.Context, extensionID string) ([]*rbacrpc.ExtensionReviewLog, error) {
	q := common.PSQL.Select(columns...).From(TableName).
		Where("extension_id = ?", extensionID).
		OrderBy("created_at DESC").
		Limit(500) // no pagination for now, just return a long list

	var list []ExtensionReviewLog
	err := d.DBX.LoadAll(ctx, &list, q)
	return ListToRPC(list), err
}

func (d *ExtensionReviewLogsDBX) GetLastInReview(ctx context.Context, extensionID, version string) (*rbacrpc.ExtensionReviewLog, error) {
	q := common.PSQL.
		Select(columns...).
		From(TableName).
		Where(squirrel.Eq{
			"extension_id":      extensionID,
			"extension_version": version,
			"state":             "review",
		}).
		OrderBy("created_at DESC").Limit(1) // latest

	var e ExtensionReviewLog
	err := d.DBX.LoadOne(ctx, &e, q)
	return e.ToRPC(), err
}

func (d *ExtensionReviewLogsDBX) GetLastWithAuthorNotes(ctx context.Context, authorTwitchID string, extensionID string) (*rbacrpc.ExtensionReviewLog, error) {
	q := common.PSQL.
		Select(columns...).
		From(TableName).
		Where(squirrel.Eq{"extension_id": extensionID, "author_twitch_id": authorTwitchID}).
		Where("author_notes <> '' OR author_channel <> ''"). // NOTE: field <> '' also excludes NULL values.
		OrderBy("created_at DESC").Limit(1)                  // latest

	var e ExtensionReviewLog
	err := d.DBX.LoadOne(ctx, &e, q)
	return e.ToRPC(), err
}

//
// Converters
//

func (e ExtensionReviewLog) ToRPC() *rbacrpc.ExtensionReviewLog {
	return &rbacrpc.ExtensionReviewLog{
		CreatedAt:         e.CreatedAt,
		State:             e.State,
		AuthorTwitchId:    e.AuthorTwitchId,
		AuthorNotes:       e.AuthorNotes,
		AuthorChannel:     e.AuthorChannel,
		ExtensionId:       e.ExtensionId,
		ExtensionVersion:  e.ExtensionVersion,
		ReviewerTwitchId:  e.ReviewerTwitchId,
		ReviewReasonCodes: e.ReviewReasonCodes,
		ReviewReason:      e.ReviewReason,
		SalesforceCaseId:  e.SalesforceCaseId,
	}
}

func ListToRPC(list []ExtensionReviewLog) []*rbacrpc.ExtensionReviewLog {
	out := make([]*rbacrpc.ExtensionReviewLog, len(list))
	for i, e := range list {
		out[i] = e.ToRPC()
	}
	return out
}

func ConvertFromRPC(in *rbacrpc.ExtensionReviewLog) ExtensionReviewLog {
	return ExtensionReviewLog{
		CreatedAt:         in.CreatedAt,
		State:             in.State,
		AuthorTwitchId:    in.AuthorTwitchId,
		AuthorNotes:       in.AuthorNotes,
		AuthorChannel:     in.AuthorChannel,
		ExtensionId:       in.ExtensionId,
		ExtensionVersion:  in.ExtensionVersion,
		ReviewerTwitchId:  in.ReviewerTwitchId,
		ReviewReasonCodes: in.ReviewReasonCodes,
		ReviewReason:      in.ReviewReason,
		SalesforceCaseId:  in.SalesforceCaseId,
	}
}
