package featuregating

import (
	"context"
	"database/sql"
	"errors"
	"fmt"

	"github.com/lib/pq"

	"code.justin.tv/devrel/dbx"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"code.justin.tv/devrel/devsite-rbac/packagewrapper/localcache"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"github.com/cactus/go-statsd-client/statsd"
)

// by default, if there isn't a key, the default value will be false
// make sure to match the default behavior as "false" and behaviour of new feature as "true"
// const (
// 	FEATURE_GATING_KEY_ROLE_MIGRATION = "feature_gating_key_role_migration_enabled"
// )

//go:generate counterfeiter . FeatureGating
//go:generate errxer --timings FeatureGating
type FeatureGating interface {
	BoolFeatureGate(ctx context.Context, key string) (bool, error)
	SetBoolFeatureGate(ctx context.Context, key string, value bool) error
	StringsFeatureGate(ctx context.Context, key string) ([]string, error)
	UpdateStringsFeatureGate(ctx context.Context, key string, value string, action rbacrpc.UpdateFeatureGatingValueAction) ([]string, error)
}

var (
	ErrValueAlreadyExists = errors.New("value already exists")
	ErrValueDoesntExist   = errors.New("value doesn't exist")
)

type CachedFeatureGatingImpl struct {
	Cache localcache.LocalCache
	FeatureGating
}

type backend struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter, cache localcache.LocalCache) FeatureGating {
	return &CachedFeatureGatingImpl{
		Cache: cache,
		FeatureGating: &FeatureGatingErrx{
			FeatureGating: &backend{
				db: db,
			},
			TimingFunc: common.TimingStats(stats),
		},
	}
}

func (b *backend) BoolFeatureGate(ctx context.Context, key string) (bool, error) {
	var f FeatureGatingRow
	err := b.db.LoadOne(
		ctx,
		&f,
		common.PSQL.Select(Columns...).
			From(Table).
			Where(featureGatingKeyEquation, key).Limit(1))
	if !f.FeatureGatingValueInBool.Valid {
		return false, err
	}
	return f.FeatureGatingValueInBool.Bool, err
}

func (b *backend) SetBoolFeatureGate(ctx context.Context, key string, value bool) error {
	currentValue, err := b.BoolFeatureGate(ctx, key)
	if errorutil.IsErrNoRows(err) {
		// no such key, insert a new key-value pair
		err = b.db.InsertOne(ctx, Table, dbx.Values{
			featureGatingKeyCol:         key,
			featureGatingValueInBoolCol: sql.NullBool{Bool: value, Valid: true},
		})
	} else if err == nil {
		// update the existing record when value to update is not the same as value in the DB
		if currentValue != value {
			err = b.db.UpdateOne(ctx, Table, dbx.Values{
				featureGatingKeyCol:         key,
				featureGatingValueInBoolCol: sql.NullBool{Bool: value, Valid: true},
			}, dbx.FindBy(featureGatingKeyCol))
		}
	}
	return err
}

func (b *backend) StringsFeatureGate(ctx context.Context, key string) ([]string, error) {
	var f FeatureGatingRow
	err := b.db.LoadOne(
		ctx,
		&f,
		common.PSQL.Select(Columns...).
			From(Table).
			Where(featureGatingKeyEquation, key).Limit(1))
	if f.FeatureGatingValueInStrings == nil {
		return []string{}, err
	}
	return f.FeatureGatingValueInStrings, err
}

func (b *backend) UpdateStringsFeatureGate(ctx context.Context, key string, value string, action rbacrpc.UpdateFeatureGatingValueAction) ([]string, error) {
	currentValues, fetchErr := b.StringsFeatureGate(ctx, key)
	if fetchErr != nil && !errorutil.IsErrNoRows(fetchErr) {
		return nil, fetchErr
	}
	result, err := b.getStringsToUpdate(ctx, currentValues, value, action)
	if err != nil {
		return nil, err
	}
	if errorutil.IsErrNoRows(fetchErr) {
		// if there is no record in the DB for the providing feature gating key, we need to insert a new record.
		err = b.db.InsertOne(ctx, Table, dbx.Values{
			featureGatingKeyCol:            key,
			featureGatingValueInStringsCol: pq.Array(result),
		})
	} else {
		// if there is a record already we just need to update the existing record
		err = b.db.UpdateOne(ctx, Table, dbx.Values{
			featureGatingKeyCol:            key,
			featureGatingValueInStringsCol: pq.Array(result),
		}, dbx.FindBy(featureGatingKeyCol))
	}
	if err != nil {
		return nil, nil
	}
	return result, err
}

func (b *backend) getStringsToUpdate(ctx context.Context, currentValues []string, value string, action rbacrpc.UpdateFeatureGatingValueAction) ([]string, error) {
	result := currentValues
	switch action {
	case rbacrpc.UpdateFeatureGatingValueAction_ADD:
		for _, v := range currentValues {
			if v == value {
				return nil, ErrValueAlreadyExists
			}
		}
		result = append(result, value)
	case rbacrpc.UpdateFeatureGatingValueAction_REMOVE:
		var targetValueFound bool
		for idx, v := range currentValues {
			if v == value {
				result = append(currentValues[:idx], currentValues[idx+1:]...)
				targetValueFound = true
				break
			}
		}
		if !targetValueFound {
			return nil, ErrValueDoesntExist
		}
	default:
		return nil, fmt.Errorf("case %s is not implemented", action.String())
	}
	return result, nil
}

func (cf *CachedFeatureGatingImpl) BoolFeatureGate(ctx context.Context, key string) (bool, error) {
	if valueFromCache, hit := cf.Cache.Get(key); hit {
		if value, ok := valueFromCache.(bool); ok {
			return value, nil
		}
		return false, fmt.Errorf("value of key %s is not bool type", key)
	}
	result, err := cf.FeatureGating.BoolFeatureGate(ctx, key)
	if errorutil.IsErrNoRows(err) {
		cf.Cache.SetDefault(key, false)
		return false, nil
	}
	if err != nil {
		return false, err
	}
	cf.Cache.SetDefault(key, result)
	return result, nil
}

func (cf *CachedFeatureGatingImpl) SetBoolFeatureGate(ctx context.Context, key string, value bool) error {
	if err := cf.FeatureGating.SetBoolFeatureGate(ctx, key, value); err != nil {
		return err
	}
	cf.Cache.SetDefault(key, value)
	return nil
}

func (cf *CachedFeatureGatingImpl) StringsFeatureGate(ctx context.Context, key string) ([]string, error) {
	if valueFromCache, hit := cf.Cache.Get(key); hit {
		if value, ok := valueFromCache.([]string); ok {
			return value, nil
		}
		return nil, fmt.Errorf("value of key %s is not []string type", key)
	}
	result, err := cf.FeatureGating.StringsFeatureGate(ctx, key)
	if err != nil {
		if errorutil.IsErrNoRows(err) {
			cf.Cache.SetDefault(key, nil)
			return nil, nil
		}
		return nil, err
	}
	cf.Cache.SetDefault(key, result)
	return result, nil
}

func (cf *CachedFeatureGatingImpl) UpdateStringsFeatureGate(ctx context.Context, key string, value string, action rbacrpc.UpdateFeatureGatingValueAction) ([]string, error) {
	result, err := cf.FeatureGating.UpdateStringsFeatureGate(ctx, key, value, action)
	if err != nil {
		return result, err
	}
	cf.Cache.SetDefault(key, result)
	return result, err
}
