package featuregating_test

import (
	"context"
	"testing"

	"github.com/stretchr/testify/require"

	"code.justin.tv/devrel/devsite-rbac/backend/featuregating"
	"code.justin.tv/devrel/devsite-rbac/backend/featuregating/featuregatingfakes"
	"code.justin.tv/devrel/devsite-rbac/backend/testutils"
	"code.justin.tv/devrel/devsite-rbac/packagewrapper/localcache/localcachefakes"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func TestFeatureGateBool(t *testing.T) {
	t.Run("If there is no value in local cache, DB query will be called", func(t *testing.T) {
		fakeBackender, fakeLocalCache, cachedBackender := initFakeEntities()
		ctx := testutils.MockContextWithOauthToken(context.Background())
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)

		fakeLocalCache.GetReturns(false, false)

		_, err := cachedBackender.BoolFeatureGate(ctx, "any_feature_gating_key")

		require.Nil(t, err)
		require.Equal(t, 1, fakeBackender.BoolFeatureGateCallCount())
	})

	t.Run("Value will be cached if there was no value before, then before the value is expired, DB query will not be called", func(t *testing.T) {
		fakeBackender, fakeLocalCache, cachedBackender := initFakeEntities()
		testKey := "any_feature_gating_key"
		ctx := testutils.MockContextWithOauthToken(context.Background())
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)

		fakeLocalCache.GetReturns(false, false)
		fakeLocalCache.SetDefaultStub = func(k string, value interface{}) {
			fakeLocalCache.GetReturns(value, true)
		}
		require.Equal(t, 0, fakeBackender.BoolFeatureGateCallCount())

		fakeBackender.BoolFeatureGateReturns(true, nil)

		value, err := cachedBackender.BoolFeatureGate(ctx, testKey)
		cachedValue, ok := fakeLocalCache.Get(testKey)

		require.Equal(t, true, value)
		// verify that cache is updated
		require.Equal(t, true, ok)
		require.Equal(t, true, cachedValue)
		require.Nil(t, err)
		require.Equal(t, 1, fakeBackender.BoolFeatureGateCallCount())

		value, err = cachedBackender.BoolFeatureGate(ctx, testKey)

		require.Equal(t, true, value)
		require.Nil(t, err)
		// no more DB query
		require.Equal(t, 1, fakeBackender.BoolFeatureGateCallCount())

	})
}

func TestSetBoolFeatureGate(t *testing.T) {
	t.Run("Updating feature gating value will also refresh cache", func(t *testing.T) {
		fakeBackender, fakeLocalCache, cachedBackender := initFakeEntities()
		testKey := "any_feature_gating_key"
		ctx := testutils.MockContextWithOauthToken(context.Background())
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)

		fakeLocalCache.GetReturns(false, false)
		fakeLocalCache.SetDefaultStub = func(k string, value interface{}) {
			fakeLocalCache.GetReturns(value, true)
		}

		cachedValue, ok := cachedBackender.Cache.Get(testKey)
		require.Equal(t, false, ok)
		require.Equal(t, false, cachedValue)
		require.Equal(t, 0, fakeBackender.BoolFeatureGateCallCount())

		err := cachedBackender.SetBoolFeatureGate(ctx, testKey, true)

		require.Nil(t, err)
		require.Equal(t, 1, fakeBackender.SetBoolFeatureGateCallCount())

		// verify that cache is refreshed
		cachedValue, ok = cachedBackender.Cache.Get(testKey)
		require.Equal(t, true, ok)
		require.Equal(t, true, cachedValue)
	})
}

func TestFeatureGateStrings(t *testing.T) {
	t.Run("If there is no value in local cache, DB query will be called", func(t *testing.T) {
		fakeBackender, fakeLocalCache, cachedBackender := initFakeEntities()
		ctx := testutils.MockContextWithOauthToken(context.Background())
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)

		fakeLocalCache.GetReturns(nil, false)

		_, err := cachedBackender.StringsFeatureGate(ctx, "any_feature_gating_key")

		require.Nil(t, err)
		require.Equal(t, 1, fakeBackender.StringsFeatureGateCallCount())
	})

	t.Run("Value will be cached if there was no value before, then before the value is expired, DB query will not be called", func(t *testing.T) {
		fakeBackender, fakeLocalCache, cachedBackender := initFakeEntities()
		testKey := "any_feature_gating_key"
		ctx := testutils.MockContextWithOauthToken(context.Background())
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)

		fakeLocalCache.GetReturns(nil, false)
		fakeLocalCache.SetDefaultStub = func(k string, value interface{}) {
			fakeLocalCache.GetReturns(value, true)
		}
		require.Equal(t, 0, fakeBackender.StringsFeatureGateCallCount())

		fakeBackender.StringsFeatureGateReturns([]string{"test_fg_1", "test_fg_2"}, nil)

		value, err := cachedBackender.StringsFeatureGate(ctx, testKey)
		cachedValue, ok := fakeLocalCache.Get(testKey)
		require.Equal(t, true, ok)
		castedCachedValue, ok := cachedValue.([]string)

		require.Equal(t, 2, len(value))
		require.Equal(t, "test_fg_1", value[0])
		require.Equal(t, "test_fg_2", value[1])
		// verify that cache is updated
		require.Equal(t, true, ok)
		require.Equal(t, "test_fg_1", castedCachedValue[0])
		require.Equal(t, "test_fg_2", castedCachedValue[1])
		require.Nil(t, err)
		require.Equal(t, 1, fakeBackender.StringsFeatureGateCallCount())

		value, err = cachedBackender.StringsFeatureGate(ctx, testKey)

		require.Equal(t, 2, len(value))
		require.Equal(t, "test_fg_1", value[0])
		require.Equal(t, "test_fg_2", value[1])
		require.Nil(t, err)
		// no more DB query
		require.Equal(t, 1, fakeBackender.StringsFeatureGateCallCount())
	})
}

func TestUpdateStringsFeatureGate(t *testing.T) {
	t.Run("Updating feature gating value will also refresh cache", func(t *testing.T) {
		fakeBackender, fakeLocalCache, cachedBackender := initFakeEntities()
		testKey := "any_feature_gating_key"
		ctx := testutils.MockContextWithOauthToken(context.Background())
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)

		fakeLocalCache.SetDefaultStub = func(k string, value interface{}) {
			fakeLocalCache.GetReturns(value, true)
		}
		fakeBackender.UpdateStringsFeatureGateReturns([]string{"test_fg_1", "test_fg_2", "test_fg_3"}, nil)

		cachedValue, ok := cachedBackender.Cache.Get(testKey)
		require.Equal(t, false, ok)
		require.Equal(t, nil, cachedValue)
		require.Equal(t, 0, fakeBackender.UpdateStringsFeatureGateCallCount())

		updatedValue, err := cachedBackender.UpdateStringsFeatureGate(ctx, testKey, "test_fg_3", rbacrpc.UpdateFeatureGatingValueAction_ADD)

		require.Nil(t, err)
		require.Equal(t, 1, fakeBackender.UpdateStringsFeatureGateCallCount())

		// verify that cache is refreshed
		cachedValue, ok = cachedBackender.Cache.Get(testKey)
		require.Equal(t, true, ok)
		castedCachedValue, ok := cachedValue.([]string)
		require.Equal(t, true, ok)
		require.Equal(t, 3, len(castedCachedValue))
		require.Equal(t, updatedValue[0], castedCachedValue[0])
		require.Equal(t, updatedValue[1], castedCachedValue[1])
		require.Equal(t, updatedValue[2], castedCachedValue[2])
	})
}

func initFakeEntities() (*featuregatingfakes.FakeFeatureGating, *localcachefakes.FakeLocalCache, *featuregating.CachedFeatureGatingImpl) {
	fakeBackender := &featuregatingfakes.FakeFeatureGating{}
	fakeLocalCache := &localcachefakes.FakeLocalCache{}
	cachedBackender := &featuregating.CachedFeatureGatingImpl{
		Cache:         fakeLocalCache,
		FeatureGating: fakeBackender,
	}
	return fakeBackender, fakeLocalCache, cachedBackender
}
