package memberships

import (
	"context"
	"database/sql"
	"fmt"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	rcommon "code.justin.tv/devrel/devsite-rbac/common"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

const (
	Table                 = "memberships"
	MaxAllowedMemberships = 100 // If set to 0 the check will not be performed, set to 1 to go dark with multiorg
	MaxStaffMemberships   = 5000
)

//go:generate counterfeiter . Memberships
//go:generate errxer --timings Memberships
type Memberships interface {
	GetMembership(ctx context.Context, companyID, twitchID string) (Membership, error)
	ListMemberships(ctx context.Context, params ListMembershipsParams) ([]Membership, int32, error)

	InsertMembership(ctx context.Context, m *Membership) error
	UpdateMembership(ctx context.Context, m *Membership) error
	DeleteMembership(ctx context.Context, companyID, twitchID string) error
	GetAdminsByCompany(ctx context.Context, companyID string) ([]Membership, error)
	GetMembershipsByAccount(ctx context.Context, twitchID string) ([]Membership, error)
	MigrateOwner(ctx context.Context, companyID string, oldOwnerID string, newOwnerID string) error
	DeleteCompanyMemberships(ctx context.Context, companyID string) error
}

type Membership struct {
	CompanyID string `db:"company_id"`
	TwitchID  string `db:"twitch_id"`

	Role string `db:"role"`

	FirstName string `db:"first_name"`
	LastName  string `db:"last_name"`
	DevTitle  string `db:"dev_title"`
	DevEmail  string `db:"dev_email"`

	CreatedAt  string         `db:"created_at"`
	ModifiedAt sql.NullString `db:"modified_at"`

	XXX_Total int32 `db:"_total"`
}

var Columns = dbx.FieldsFrom(&Membership{}).Exclude("_total", "user_id")

type DBXMemberships struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter) Memberships {
	impl := &DBXMemberships{db: db}
	errxWrap := &MembershipsErrx{
		Memberships: impl,
		TimingFunc:  common.TimingStats(stats),
	}
	return errxWrap
}

func (d *DBXMemberships) GetMembership(ctx context.Context, companyID, twitchID string) (Membership, error) {
	q := common.PSQL.Select(Columns...).From(Table).
		Where("company_id = ? AND twitch_id = ?", companyID, twitchID).Limit(1)
	var m Membership
	err := d.db.LoadOne(ctx, &m, q)
	return m, err
}

type ListMembershipsParams struct {
	CompanyID string
	TwitchID  string
	Role      string

	Limit      uint64
	Offset     uint64
	OrderBy    string // field to order
	OrderByDir string // DESC or ASC (default)
}

func (d *DBXMemberships) ListMemberships(ctx context.Context, p ListMembershipsParams) ([]Membership, int32, error) {
	cols := Columns.Add(common.CountOverAs("_total"))
	q := common.PSQL.Select(cols...).From(Table)

	if p.CompanyID != "" {
		q = q.Where("company_id = ?", p.CompanyID)
	}
	if p.TwitchID != "" {
		q = q.Where("twitch_id = ?", p.TwitchID)
	}
	if p.Role != "" {
		q = q.Where("role = ?", p.Role)
	}

	q = common.Paginate(q, p.Limit, p.Offset)
	if p.OrderBy != "" {
		q = q.OrderBy(p.OrderBy + " " + p.OrderByDir)
	}

	list := []Membership{}
	err := d.db.LoadAll(ctx, &list, q)
	return list, common.FirstRowInt32DBField(list, "_total"), err
}

func (d *DBXMemberships) InsertMembership(ctx context.Context, m *Membership) error {
	m.CreatedAt = common.TimeNowStr()
	return d.db.InsertOne(ctx, Table, m, dbx.Exclude("_total"))
}

func (d *DBXMemberships) UpdateMembership(ctx context.Context, m *Membership) error {
	m.ModifiedAt = common.NewSQLNullString(common.TimeNowStr())
	return d.db.UpdateOne(ctx, Table, m, dbx.FindBy("company_id", "twitch_id"),
		dbx.Exclude("_total", "created_at"))
}

func (d *DBXMemberships) DeleteMembership(ctx context.Context, companyID, twitchID string) error {
	return d.db.DeleteOne(ctx, Table, dbx.Values{"company_id": companyID, "twitch_id": twitchID})
}

func (d *DBXMemberships) GetAdminsByCompany(ctx context.Context, companyID string) ([]Membership, error) {
	q := common.PSQL.Select(Columns...).From(Table).
		Where("company_id = ? AND role = ?", companyID, rcommon.AdminRole).OrderBy("created_at DESC")
	var m []Membership
	err := d.db.LoadAll(ctx, &m, q)
	return m, err
}

func (d *DBXMemberships) GetMembershipsByAccount(ctx context.Context, twitchID string) ([]Membership, error) {
	q := common.PSQL.Select(Columns...).From(Table).
		Where("twitch_id = ?", twitchID)
	var m []Membership
	err := d.db.LoadAll(ctx, &m, q)
	return m, err
}

func (d *DBXMemberships) MigrateOwner(ctx context.Context, companyID string, oldOwnerID string, newOwnerID string) error {
	oldUser := Membership{
		TwitchID:  oldOwnerID,
		CompanyID: companyID,
		Role:      rcommon.AdminRole,
	}

	newUser := Membership{
		TwitchID:  newOwnerID,
		CompanyID: companyID,
		Role:      rcommon.OwnerRole,
	}

	err := d.db.UpdateOne(ctx, Table, &newUser, dbx.FindBy("company_id", "twitch_id"), dbx.Exclude("company_id", "twitch_id", "first_name", "last_name", "dev_title", "dev_email", "created_at", "modified_at", "_total"))
	if err != nil {
		return err
	}

	err = d.db.UpdateOne(ctx, Table, &oldUser, dbx.FindBy("company_id", "twitch_id"), dbx.Exclude("company_id", "twitch_id", "first_name", "last_name", "dev_title", "dev_email", "created_at", "modified_at", "_total"))
	if err != nil {
		return err
	}

	return nil
}

func (d *DBXMemberships) DeleteCompanyMemberships(ctx context.Context, companyID string) error {
	_, err := d.db.NamedExec(ctx, fmt.Sprintf("DELETE FROM %s WHERE company_id = :company_id", Table), Membership{
		CompanyID: companyID,
	})

	return err
}

//
// Converters
//

func (m Membership) ToRPC() *rbacrpc.Membership {
	return &rbacrpc.Membership{
		CompanyId:  m.CompanyID,
		TwitchId:   m.TwitchID,
		Role:       m.Role,
		FirstName:  m.FirstName,
		LastName:   m.LastName,
		DevTitle:   m.DevTitle,
		DevEmail:   m.DevEmail,
		CreatedAt:  m.CreatedAt,
		ModifiedAt: m.ModifiedAt.String,
	}
}

func ListToRPC(list []Membership) []*rbacrpc.Membership {
	membs := make([]*rbacrpc.Membership, len(list))
	for i, memb := range list {
		membs[i] = memb.ToRPC()
	}
	return membs
}
