package viennauserwhitelist

import (
	"context"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/cactus/go-statsd-client/statsd"
)

const Table = "user_whitelist"

//go:generate counterfeiter . UserWhitelist
//go:generate errxer --timings UserWhitelist
type UserWhitelist interface {
	// List whitelisted users by role
	ListUserWhitelistByRole(ctx context.Context, params *rbacrpc.ListUserWhitelistByRoleRequest) (*rbacrpc.ListUserWhitelistByRoleResponse, error)
	// Get whitelisted user
	GetWhitelistedUser(ctx context.Context, twitchID string) (*rbacrpc.UserWhitelist, error)
	// Insert an user into whitelist
	WhitelistUser(ctx context.Context, params *rbacrpc.WhitelistUserRequest) error
	// Update an user's role in the whitelist
	UpdateUserRoleInWhitelist(ctx context.Context, params *rbacrpc.WhitelistUserRequest) error
	// Remove an user from whitelist
	RemoveUserFromWhitelist(ctx context.Context, params *rbacrpc.RemoveUserFromWhitelistRequest) error
}

type UserWhitelistRow struct {
	TwitchID string `db:"twitch_id"`
	Role     string `db:"role"` // Vienna access role (NOT related to RBAC membership role)

	XXX_Total int32 `db:"_total"`
}

var Columns = dbx.FieldsFrom(UserWhitelistRow{}).Exclude("_total")

type backend struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter) UserWhitelist {
	impl := &backend{db: db}
	errxWrap := &UserWhitelistErrx{
		UserWhitelist: impl,
		TimingFunc:    common.TimingStats(stats),
	}
	return errxWrap
}

func (b *backend) ListUserWhitelistByRole(ctx context.Context, params *rbacrpc.ListUserWhitelistByRoleRequest) (*rbacrpc.ListUserWhitelistByRoleResponse, error) {
	q := common.PSQL.Select(common.CountOverAs("_total"), "twitch_id", "role").From(Table)
	q = q.Where("role = ?", params.Role.String())
	q = common.Paginate(q, uint64(params.Limit), uint64(params.Offset))

	var list []UserWhitelistRow
	err := b.db.LoadAll(ctx, &list, q)
	if err != nil {
		return nil, err
	}

	return &rbacrpc.ListUserWhitelistByRoleResponse{
		Users: ListToRPC(list),
		Total: common.FirstRowInt32DBField(&list, "_total"),
	}, nil
}

func (b *backend) GetWhitelistedUser(ctx context.Context, twitchID string) (*rbacrpc.UserWhitelist, error) {
	var u UserWhitelistRow
	err := b.db.LoadOne(ctx, &u, common.PSQL.Select(Columns...).From(Table).
		Where("twitch_id = ?", twitchID).Limit(1))
	return u.ToRPC(), err
}

func (b *backend) WhitelistUser(ctx context.Context, params *rbacrpc.WhitelistUserRequest) error {
	return b.db.InsertOne(ctx, Table, dbx.Values{
		"twitch_id": params.TwitchId,
		"role":      params.Role.String(),
	})
}

func (b *backend) UpdateUserRoleInWhitelist(ctx context.Context, params *rbacrpc.WhitelistUserRequest) error {
	return b.db.UpdateOne(ctx, Table, dbx.Values{
		"twitch_id": params.TwitchId,
		"role":      params.Role.String(),
	}, dbx.FindBy("twitch_id"))
}

func (b *backend) RemoveUserFromWhitelist(ctx context.Context, params *rbacrpc.RemoveUserFromWhitelistRequest) error {
	return b.db.DeleteOne(ctx, Table, dbx.Values{"twitch_id": params.TwitchId})
}

//
// Converters
//

func (u UserWhitelistRow) ToRPC() *rbacrpc.UserWhitelist {
	return &rbacrpc.UserWhitelist{
		TwitchId: u.TwitchID,
		Role:     rbacrpc.WhitelistUserRole(rbacrpc.WhitelistUserRole_value[u.Role]),
	}
}

func ListToRPC(list []UserWhitelistRow) []*rbacrpc.UserWhitelist {
	users := make([]*rbacrpc.UserWhitelist, len(list))
	for i, u := range list {
		users[i] = u.ToRPC()
	}
	return users
}
