package cartman

import (
	"context"
	"net/http"
	"net/url"

	"code.justin.tv/foundation/twitchclient"
)

// Client is a simplified interface for the Cartman service.
//go:generate errxer Client
type Client interface {

	// Get a JWT Cartman Token to use on calls to services that need the Twitch-Authorization header.
	// Used for EMS and Moneypenny services.
	// Most services called from GraphQL use the capability "cartman::authenticate_first_party".
	// If more capabilities are needed, they need to be comma separated, for example:
	//   "extensions::view_all_extensions,extensions::edit_version,extensions::review_extensions".
	// The EMS service requires different capabilities for different endpoints, see:
	//   https://git-aws.internal.justin.tv/edge/visage/blob/b90b6cae39cf8972408aa2c4bec2b8b844c22a92/api/v5/devsuccess/extensions/module.go#L16)
	// The authHeader is the OAuth token (e.g. "Bearer 123abcdf") used by Cartman to identify the current user.
	// The authFnParams are extra parameters for specific authorization functions, depending on the capability.
	// For example, "extensions::view_all_extensions" requires an "extension_id" parameter.
	// If no authFnParams are required, nil can be used.
	GetToken(ctx context.Context, authHeader, capabilities string, authFnParams map[string]string) (string, error)
}

func NewClient(clientConf twitchclient.ClientConf) Client {
	httpCli := twitchclient.NewHTTPClient(clientConf)
	jsonCLi := twitchclient.NewJSONClient(clientConf.Host, httpCli)
	client := &client{JSONClient: jsonCLi}
	return &ClientErrx{Client: client}
}

type client struct {
	JSONClient *twitchclient.JSONClient
}

func (c *client) GetToken(ctx context.Context, authHeader, capabilities string, authFnParams map[string]string) (string, error) {
	respBody := tokenResponse{}

	ctx = twitchclient.WithReqOpts(ctx, twitchclient.ReqOpts{
		StatName:       "service.cartman.authorization_token",
		StatSampleRate: 1,
	})

	params := url.Values{}
	params.Add("key", "ecc.key")
	params.Add("capabilities", capabilities)
	for k, v := range authFnParams { // extra params for specific capabilities
		params.Add(k, v)
	}

	headers := http.Header{}
	headers.Add("Authorization", authHeader)
	headers.Add("Cache-Control", "no-cache")

	_, err := c.JSONClient.Do(ctx, twitchclient.JSONRequest{
		Method: "GET",
		Path:   "/authorization_token",
		Query:  params,
		Header: headers,
	}, &respBody)
	if err != nil {
		return "", err
	}

	return respBody.Token, nil
}

type tokenResponse struct {
	Token string `json:"token"`
	// UserID   string `json:"user_id"`
	// ClientID string `json:"client_id"`
}
