package clue

import (
	"context"
	"sync"

	clueCli "code.justin.tv/chat/tmi/clue/proto/clue"
	"code.justin.tv/devrel/devsite-rbac/internal/utils"
	"code.justin.tv/foundation/twitchclient"
	"github.com/hashicorp/go-multierror"
	"github.com/pkg/errors"
)

//go:generate errxer Clue
//go:generate counterfeiter . Clue
type Clue interface {
	UpdateAllowedChattersForChannels(ctx context.Context, chatterIDsToUpdate []string, channelID []string) error
}

type clueImpl struct {
	client clueCli.Clue
}

func NewClient(hostURL string, clientConf twitchclient.ClientConf) Clue {
	client := clueCli.NewClueProtobufClient(hostURL, twitchclient.NewHTTPClient(clientConf))
	return &ClueErrx{Clue: &clueImpl{client: client}}
}

func (c *clueImpl) UpdateAllowedChattersForChannels(ctx context.Context, chatterIDsToUpdate []string, channelIDs []string) error {
	wg := &sync.WaitGroup{}
	mutex := &sync.Mutex{}
	errs := make(map[string]error, len(channelIDs))
	for _, cID := range channelIDs {
		wg.Add(1)
		go func(channelID string) {
			defer wg.Done()
			if err := utils.WithRetry(func() error {
				_, err := c.updateAllowedChattersByChannelID(ctx, chatterIDsToUpdate, channelID)
				return err
			}, 3); err != nil {
				mutex.Lock()
				errs[channelID] = err
				mutex.Unlock()
			}
		}(cID)
	}
	wg.Wait()

	var multierr *multierror.Error
	for id, err := range errs {
		multierr = multierror.Append(multierr, errors.Wrap(err, "failed to update allowed chatters for channel: "+id))
	}

	return multierr.ErrorOrNil()
}

func (c *clueImpl) updateAllowedChattersByChannelID(ctx context.Context, chatterIDsToUpdate []string, channelID string) ([]string, error) {
	req := &clueCli.UpdateAllowedChatterIDsRequest{
		ChannelId:         channelID,
		AllowedChatterIds: chatterIDsToUpdate,
	}
	resp, err := c.client.UpdateAllowedChatterIDs(ctx, req)
	if err != nil {
		return nil, err
	}
	return resp.UpdatedAllowedChatterIds, nil
}
