package discovery

import (
	"context"
	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/foundation/twitchclient"
	dclient "code.justin.tv/web/discovery/client"
)

const getAllBatchSize = 50

// Discovery client is a simplified client for the Discovery service
//go:generate errxer Discovery
//go:generate counterfeiter . Discovery
type Discovery interface {
	GetAll(ctx context.Context, gameIDs []string) (map[string]*dclient.LocalizedGame, error)
}

type discoveryImpl struct {
	client dclient.Client
	stats  statter
}

type statter interface {
	Inc(stat string, count int64, rate float32) error
}

func NewDiscovery(clientConf twitchclient.ClientConf) (Discovery, error) {
	client, err := dclient.NewClient(clientConf)
	if err != nil {
		return nil, err
	}
	return &DiscoveryErrx{Discovery: &discoveryImpl{client: client, stats: clientConf.Stats}}, nil
}

func (d *discoveryImpl) GetAll(ctx context.Context, gameIDs []string) (map[string]*dclient.LocalizedGame, error) {
	games := map[string]*dclient.LocalizedGame{}
	idsBatches := splitInBatches(gameIDs, getAllBatchSize)
	for i, idsBatch := range idsBatches {
		resp, err := d.client.GetAll(ctx, idsBatch, "", getAllBatchSize, 0, defaultReqOpts())
		if err != nil {
			status := errorutil.StatusCode(err)
			_ = d.stats.Inc(fmt.Sprintf("service.discovery.GetAll.%d", status), 1, 1)
			return nil, errx.Wrap(err, fmt.Sprintf("discovery.GetAll: status %d: loading batch %d of %d (batch size: %d gameIDs)", status, i+1, len(idsBatches), len(idsBatch)))
		}
		_ = d.stats.Inc("service.discovery.GetAll.200", 1, 1)
		for gameID, game := range resp {
			games[gameID] = game
		}
	}
	return games, nil
}

func splitInBatches(gameIDs []string, batchSize int) [][]string {
	batches := [][]string{}
	for i := 0; i < len(gameIDs); i += batchSize {
		end := i + batchSize
		if end > len(gameIDs) {
			end = len(gameIDs)
		}
		batches = append(batches, gameIDs[i:end])
	}
	return batches
}

func defaultReqOpts() *twitchclient.ReqOpts {
	return &twitchclient.ReqOpts{StatSampleRate: 1}
}
