package evs

import (
	"context"
	"fmt"
	"net/mail"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"github.com/twitchtv/twirp"

	evscli "code.justin.tv/amzn/TwitchEmailValidatorServiceTwirp"
	"code.justin.tv/amzn/TwitchS2S2/s2s2"
	"code.justin.tv/foundation/twitchclient"
)

// EVS (Email Validator Service) is a simplified interface for the
// the EVS service (https://git-aws.internal.justin.tv/growth/emailvalidator)
//go:generate errxer EVS
//go:generate counterfeiter . EVS
type EVS interface {
	SendVerificationEmail(ctx context.Context, evsKey string, email string, purpose string) error
	GetVerificationStatus(ctx context.Context, evsKey string, email string) string
	DeleteVerificationStatus(ctx context.Context, evsKey string, email string) error
}

type evsImpl struct {
	client evscli.TwitchEmailValidatorService
}

func NewEVS(EVSHost string, clientConf twitchclient.ClientConf, s2s2Client *s2s2.S2S2) (EVS, error) {

	client := evscli.NewTwitchEmailValidatorServiceProtobufClient(EVSHost, s2s2Client.HTTPClient(twitchclient.NewHTTPClient(clientConf)))
	return &EVSErrx{EVS: &evsImpl{client: client}}, nil
}

func (e *evsImpl) SendVerificationEmail(ctx context.Context, evsKey string, email string, purpose string) error {
	if !isWellFormedEmail(email) {
		return fmt.Errorf("malformed email %q for evsKey %q", email, evsKey)
	}

	_, err := e.client.AddVerificationRequest(ctx, &evscli.AddVerificationRequestInput{Namespace: "developers", Key: evsKey, Email: email, Locale: "en", Purpose: purpose, ShouldIncludeCode: false})
	if err != nil {
		twerr, ok := err.(twirp.Error)
		if ok && twerr.Code() == twirp.FailedPrecondition {
			// Non-pending verification for this email already exists.
			// For us this is a noop to allow idempotency.
			return nil
		}
		return err
	}

	return nil
}

func (e *evsImpl) GetVerificationStatus(ctx context.Context, evsKey string, email string) string {
	out, err := e.client.GetVerificationRequest(ctx, &evscli.GetVerificationRequestInput{Namespace: "developers", Key: evsKey, Email: email})
	if err != nil {
		logx.Warn(ctx, err)
		return "Unknown"
	}

	return out.GetVerificationRequest().GetStatus()
}

func (e *evsImpl) DeleteVerificationStatus(ctx context.Context, evsKey string, email string) error {
	_, err := e.client.DeleteVerificationRequest(ctx, &evscli.DeleteVerificationRequestInput{Namespace: "developers", Key: evsKey, Email: email})
	return err
}

// Determines if an email is valid for sending
func isWellFormedEmail(email string) bool {
	// Our email validation disallows emails ending with a dot, e.g. "foo@gmail.com.".
	// Our email validation disallows emails with two domains in them, e.g. "foo@gmail.com@gmail.com".
	// mail.ParseAddress covers both of these cases.
	_, err := mail.ParseAddress(email)
	if err != nil {
		return false
	}

	// Our email validation disallows emails with spaces in them, e.g. "foo @ gmail.com".
	if strings.Contains(email, " ") {
		return false
	}

	return true
}
