package moneypenny

import (
	"context"

	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/foundation/twitchclient"
	moneypennycli "code.justin.tv/revenue/moneypenny/client"
)

//go:generate errxer Client
//go:generate counterfeiter . Client
type Client interface {
	// HasTIMSEnabled returns true if the twitchID started and finished payments onboarding, and has 2FA enabled.
	// Returns an error if something went wrong. Most errors are internal issues.
	// Please check if errorutils.StatusCode(err) == 403, that may happen because MoneyPenny checks back on RBAC
	// with RBAC.ValidateByTwitchID (permission.ViewPaymentOnboarding) to see if the user is allowed to know the status.
	HasTIMSEnabled(ctx context.Context, twitchID string, cartmanToken string) (bool, error)
}

type client struct {
	client moneypennycli.Client
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	c, err := moneypennycli.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &ClientErrx{
		Client: &client{client: c},
	}, nil
}

func (c *client) HasTIMSEnabled(ctx context.Context, twitchID string, cartmanToken string) (bool, error) {
	resp, err := c.client.GetDeveloperState(ctx, twitchID, &twitchclient.ReqOpts{
		AuthorizationToken: cartmanToken,
	})
	if err != nil {
		return false, err // internal error
	}

	logx.Info(ctx, "user's developer onboarding state", logx.Fields{
		"twitch_id":    twitchID,
		"invite_state": resp.CurrentState,
	})

	// https://git-aws.internal.justin.tv/revenue/moneypenny/blob/9551bae7019ffd46e1011bc04e0be64fafcb5810/app/api/developer.go#L23
	// one of: onboarded, not_onboarded, in_progress
	stateCompleted := (resp.CurrentState == "onboarded")
	return stateCompleted, nil
}
