package owlcli

import (
	"context"
	"strconv"
	"strings"

	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/owl/oauth2"
)

// fakeOwnerIDsClientIDs for e2e tests, mapping known clientIDs to their ownerIDs.
// Every new test that needs to transfer an extension needs to add a new client here,
// so that the ownerID is known for a given clientID, matching the user that must be
// in the same organization where the extension is being transferred.
var fakeOwnerIDsClientIDs = map[string]string{
	"client-id-00001": "999900001",
	"client-id-00002": "999900002",
	"client-id-00003": "999900003",
	"client-id-00004": "999900004",
}

const fakeOwnerIDDefault = "666611110" // this fake user owns everything else :P

// FakeOwnerIDForClientID is the helper that can be used in tests to access the client-id owner mapping
func FakeOwnerIDForClientID(clientID string) string {
	if ownerID, ok := fakeOwnerIDsClientIDs[clientID]; ok {
		return ownerID
	}
	return fakeOwnerIDDefault
}

// Fake is a fake owlcli.Client implementation that is used in e2e tests
type Fake struct{}

func (f *Fake) Validate(ctx context.Context, token string) (*oauth2.Authorization, error) {
	// Assume all non-empty tokens are valid
	if token != "" {
		return &oauth2.Authorization{
			OwnerID:   token,
			ExpiresIn: 100,
			Valid:     true,
		}, nil
	}
	return &oauth2.Authorization{
		OwnerID:   "",
		ExpiresIn: 0,
		Valid:     false,
	}, nil
}

func (f *Fake) GetClient(ctx context.Context, clientID string) (*oauth2.Client, error) {
	if strings.HasPrefix(clientID, "invalid") {
		return nil, owl.ErrInvalidClientID // "invalid" prefix forces this error in tests
	}

	ownerID := fakeOwnerIDForClientIDInt(clientID) // return a known ownerID if available

	return &oauth2.Client{
		ClientID: clientID,
		OwnerID:  &ownerID,
	}, nil
}

func (f *Fake) UpdateClientOwner(ctx context.Context, params UpdateClientRequest) error {
	if strings.HasPrefix(params.ClientID, "invalid") {
		return owl.ErrInvalidClientID // "invalid" prefix forces this error in tests
	}
	return nil
}

//
// Helpers
//

func fakeOwnerIDForClientIDInt(clientID string) int {
	ownerID := FakeOwnerIDForClientID(clientID)
	ownerIDint64, err := strconv.ParseInt(ownerID, 10, 32)
	if err != nil {
		panic(err) // if we specify a non-numeric string id, this will show the stacktrace in the tests
	}
	return int(ownerIDint64)
}
