package pdms

import (
	"code.justin.tv/amzn/PDMSLambdaTwirp"
	"context"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/golang/protobuf/ptypes/timestamp"
	"net/http"
	"time"

	twirplambdatransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
)

//go:generate errxer PDMS
//go:generate counterfeiter . PDMS
type PDMS interface {
	GetOutstandingDeletionRequests(ctx context.Context) ([]string, error)
	MarkDeleted(ctx context.Context, userId string) error
}

type pdmsClient struct {
	client PDMSLambdaTwirp.PDMSService
}

func NewClient(PdmsCallerRole string, PdmsLambdaArn string) PDMS {
	sess := session.Must(session.NewSessionWithOptions(session.Options{
		Config: aws.Config{
			Region:              aws.String("us-west-2"),
			STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
			// We want a long timeout for PDMS, as Lambda based services can take a while to warm up on cold start.
			HTTPClient: &http.Client{Timeout: 10 * time.Second},
		},
	}))
	creds := stscreds.NewCredentials(sess, PdmsCallerRole)
	pdmsLambdaCli := lambda.New(sess, &aws.Config{Credentials: creds})
	pdmsTransport := twirplambdatransport.NewClient(pdmsLambdaCli, PdmsLambdaArn)
	return &PDMSErrx{PDMS: &pdmsClient{client: PDMSLambdaTwirp.NewPDMSServiceProtobufClient("", pdmsTransport)}}
}

func (c *pdmsClient) GetOutstandingDeletionRequests(ctx context.Context) ([]string, error) {
	req := &PDMSLambdaTwirp.GetOutstandingDeletionRequestsByServiceRequest{
		ServiceId: "448", // RBAC's service id
		CreatedBefore: &timestamp.Timestamp{
			Seconds: time.Now().Unix(),
		},
		CreatedAfter: &timestamp.Timestamp{
			Seconds: time.Now().AddDate(-1, 0, 0).Unix(),
		},
	}
	result, err := c.client.GetOutstandingDeletionRequestsByService(ctx, req)

	if err != nil {
		return nil, err
	}

	var userIds []string
	for _, element := range result.PendingDeletionRequests {
		userIds = append(userIds, element.UserId)
	}

	return userIds, nil
}

func (c *pdmsClient) MarkDeleted(ctx context.Context, userId string) error {
	req := &PDMSLambdaTwirp.ReportDeletionRequest {
		ServiceId: "448", // RBAC's service id
		UserId: userId,
	}

	_, err := c.client.ReportDeletion(ctx, req)

	return err
}
