package salesforce

import (
	"context"
	"errors"
	"io/ioutil"
	"net/http"
	"strings"
	"testing"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/require"
)

func TestNewSalesforceClient(t *testing.T) {
	config := validConfig()
	c, err := NewSalesforceClient(config)
	require.NotNil(t, c)
	require.NoError(t, err)

	// Returns an error if the AuthURL is empty
	config = validConfig()
	config.Host = ""
	c, err = NewSalesforceClient(config)
	require.NotNil(t, c)
	require.EqualError(t, err, "twirp error invalid_argument: Config.Host is required")

	// Returns if other values are empty
	config = validConfig()
	config.ClientID = ""
	c, err = NewSalesforceClient(config)
	require.NotNil(t, c)
	require.EqualError(t, err, "twirp error invalid_argument: Config.ClientID is required")
}

func TestCreateExtensionReviewCase_NoSuccessWithErrors(t *testing.T) {
	httpCli := fakeHTTPClient{status: 201, respBody: `{
		"id": "",
		"errors": ["bad parameters"],
		"success": false
	}`}
	c := newInitializedClient(t, &httpCli)

	_, err := c.CreateExtensionReviewCase(ctx, validExtensionReviewCase())
	require.EqualError(t, err, "Salesforce API: CreateExtensionReviewCase: response success: false, errors: [bad parameters]")
}

func TestCreateExtensionReviewCase_InternalError(t *testing.T) {
	httpCli := fakeHTTPClient{status: 500, respBody: `epic-fail`}
	c := newInitializedClient(t, &httpCli)

	_, err := c.CreateExtensionReviewCase(ctx, validExtensionReviewCase())
	require.EqualError(t, err, "Salesforce API: CreateExtensionReviewCase: response status 500: epic-fail")
}

func TestCreateExtensionReviewCase_NetworkError(t *testing.T) {
	httpCli := fakeHTTPClient{err: errors.New("Fake network error")}
	c := newInitializedClient(t, &httpCli)

	_, err := c.CreateExtensionReviewCase(ctx, validExtensionReviewCase())
	require.EqualError(t, err, "Fake network error")
}

// ----------
// Test Utils
// ----------

var ctx = context.Background()

func validConfig() Config {
	return Config{
		Host:          "https://config.Host.tv",
		ClientID:      "test_client_id",
		Username:      "test_username",
		RSA256PrivKey: rsa256Pem,

		Stats: &statsd.NoopClient{},
	}
}

func validExtensionReviewCase() ExtensionReviewCase {
	return ExtensionReviewCase{
		Subject:           "test-Subject",
		Origin:            "test-Origin",
		Description:       "test-Description",
		ContactEmail:      "test-ContactEmail@email.com",
		SuppliedEmail:     "test-SuppliedEmail@email.com",
		TwitchUserID:      "test-TwitchUserID__c",
		TwitchLogin:       "test-TwitchUsername__c",
		ExtensionID:       "test-Extension_ID__c",
		ExtensionVersion:  "test-Extension_Version__c",
		TwitchTestChannel: "test-Twitch_Test_Channel__c",
	}
}

func newInitializedClient(t *testing.T, httpClient *fakeHTTPClient) *client {
	return &client{
		Config:     validConfig(),
		HTTPClient: httpClient,

		InstanceURL:        "http://test",
		AccessToken:        "testtoken",
		AccessTokenExpires: time.Now().Add(1 * time.Hour), // not expired, no need to call authenticate on every test
	}
}

// fakeHTTPClient can be configured to fake different status and errors
type fakeHTTPClient struct {
	status   int    // response status, default 200.
	respBody string // response body as string.
	err      error  // response HTTP error, default nil.

	sentReq *http.Request // can be used after calling .Do for inspection.
}

// Do makes a fake request that depends on the configured fake attrs.
func (c *fakeHTTPClient) Do(req *http.Request) (*http.Response, error) {
	c.sentReq = req

	if c.err != nil {
		return nil, c.err
	}

	if c.status == 0 {
		c.status = 200
	}

	body := ioutil.NopCloser(strings.NewReader(c.respBody))
	return &http.Response{StatusCode: c.status, Body: body}, nil
}
