package users

import (
	"context"
	"math/rand"
	"strconv"
	"time"

	"code.justin.tv/web/users-service/models"
)

const (
	MockedDeletedUserTwitchID = "12213438"
	MockedIsEmailVerified     = true
)

// NoopClient implements the Users interface as a NOOP,
// to provide predictable results in a test environment.
type NoopClient struct {
	GetUserByIDReturnsError error
	GetStaffUser            bool
}

var currentTime = time.Now()
var emailVerified bool = MockedIsEmailVerified

// todo: is there an elegant way instead of hard-code?
var fakeUserProperties = map[string]*models.Properties{
	MockedDeletedUserTwitchID: &models.Properties{
		ID:            MockedDeletedUserTwitchID,
		DeletedOn:     &currentTime,
		EmailVerified: &emailVerified,
	},
}

// GetUserByID does nothing
func (c *NoopClient) GetUserByID(ctx context.Context, twitchID string) (*models.Properties, error) {
	if c.GetUserByIDReturnsError != nil {
		return nil, c.GetUserByIDReturnsError
	}

	if mockedUser, ok := fakeUserProperties[twitchID]; ok {
		return mockedUser, nil
	}

	username := "test"
	email := "test@email.com"
	emailVerified := true
	isStaff := c.GetStaffUser

	return &models.Properties{
		ID:            twitchID,
		Login:         &username,
		Email:         &email,
		EmailVerified: &emailVerified,
		Admin:         &isStaff,
	}, nil
}

// GetUserByLogin does nothing
func (c *NoopClient) GetUserByLogin(ctx context.Context, login string) (*models.Properties, error) {
	if c.GetUserByIDReturnsError != nil {
		return nil, c.GetUserByIDReturnsError
	}

	rand.Seed(time.Now().UnixNano())
	randomNum := random(10000000, 30000000)

	email := "test@email.com"
	return &models.Properties{
		ID:    strconv.FormatInt(randomNum, 10),
		Login: &login,
		Email: &email,
	}, nil
}

func (c *NoopClient) IsValidTwitchID(ctx context.Context, twitchID string) (bool, error) {
	return true, nil
}

func random(min int64, max int64) int64 {
	return rand.Int63n(max-min) + min
}
