// Generate a Salesforce OAuth JWT for development-use.
// The token is used to access the Salesforce API (e.g. query).
// https://help.salesforce.com/articleView?id=remoteaccess_oauth_jwt_flow.htm&type=5

package main

import (
	"flag"
	"fmt"
	"io/ioutil"
	"time"

	"code.justin.tv/devrel/devsite-rbac/internal/jwt"
)

var (
	audience = flag.String("audience", "https://test.salesforce.com", "Salesforce authentication url")
	issuer   = flag.String("issuer", "", "Salesforce OAuth Client ID, e.g. '3MVG973QtA4.tpvk5B__0giGWvLtLAB.4EOhUWBtsyo1tMSG5hJilydqgM4xg.qs5q.slXArNpWSi51jZLh.m'")
	subject  = flag.String("subject", "", "Username of Salesforce user, e.g. 'devportal@twitch.tv.integration.yourname'")
	keypath  = flag.String("keypath", "", "Path to the private key for signing the salesforce oauth JWT. You can generate one with ssh-keygen -t rsa.")
)

func init() {
	flag.Parse()
}

func main() {
	privateKey, err := ioutil.ReadFile(*keypath)
	if err != nil {
		panic(err)
	}

	inOneHour := time.Now().Add(1 * time.Hour)

	encoded, err := jwt.EncodeRS256(privateKey, jwt.Claims{
		Audience: *audience,
		Expires:  inOneHour.Unix(),
		Issuer:   *issuer,
		Subject:  *subject,
	})
	if err != nil {
		panic(err)
	}

	fmt.Printf("Encoded JWT (expires at %s):\n", inOneHour)
	fmt.Println(string(encoded))
}
