package config

import (
	"encoding/base64"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"reflect"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/devrel/devsite-rbac/internal/utils"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	toml "github.com/pelletier/go-toml"
)

// RBAC configures the settings for the service
type RBAC struct {
	Environment string

	RPCHostPort    string
	RollbarToken   string
	StatsdHostPort string

	SecretPrefix string

	S2SService string

	PGHost       string
	PGPort       string
	PGUser       string
	PGPassword   string
	PGDBName     string
	PGSSLEnabled bool

	SalesforceURL           string
	SalesforceUsername      string
	SalesforceClientID      string
	SalesforceRSA256PrivKey string

	DropsJWTKey string
	PdmsApiKey string
	PdmsCallerRole string
	PdmsLambdaArn string

	CartmanHost       string
	DiscoveryHost     string
	EVSHost           string
	DartHost          string
	ExtensionsEMSHost string
	HistoryHost       string
	OwlHost           string
	PushySNSTopic     string
	UsersHost         string
	PassportHost      string
	MoneypennyHost    string
	ClueHost          string
	NiohHost          string

	LocalCacheExpirationTimeInSec  int64
	LocalCacheCleanUpIntervalInSec int64

	KMSClientARN    string
	KMSClientRegion string
}

// MustLoadFromFile uses the ENVIRONMENT env var to decide what confing TOML file to load,
// using the convention: config/env-$ENVIRONMENT.toml
func MustLoadFromFile() *RBAC {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("Missing ENVIRONMENT env variable. For example ENVIRONEMT=staging")
	}
	confFilePath := "config/env-" + env + ".toml" // e.g. config/env-staging.toml
	log.Println("Env:", env)
	log.Println("Configure with", confFilePath)

	conf, err := Load(confFilePath)
	if err != nil {
		log.Fatal("Error loading conf from " + confFilePath + ": " + err.Error())
	}

	return conf
}

// Load makes a new *RBAC struct from the given TOML configuration file.
// Then loads secrets from AWS Secrets Manager if enabled in the configuration.
// Returns an error if the file could not be loaded, or parsed as TOML, or secrets failed.
func Load(filePath string) (*RBAC, error) {
	tree, err := toml.LoadFile(filePath)
	if err != nil {
		return nil, err
	}

	conf := &RBAC{}

	// Read the extended file if available
	if extendsFilePath, ok := tree.Get("EXTENDS").(string); ok {
		conf, err = Load(extendsFilePath) // load the reference first
		if err != nil {
			return nil, err
		}
	}

	// Update values present in the conf files.
	// Other values are left with default or previous extended values.
	keys := tree.Keys()
	for _, key := range keys {
		if err := conf.Set(key, tree.Get(key)); err != nil {
			return nil, err
		}
	}

	// Update values with secrets prefixes with the secrets
	err = conf.SetSecrets()
	if err != nil {
		return nil, err
	}

	return conf, nil
}

func (conf *RBAC) Set(key string, val interface{}) error {
	// EXTENDS is reserved and cannot be set
	if key == "EXTENDS" {
		return nil
	}
	field := reflect.ValueOf(conf).Elem().FieldByName(key)
	if !field.IsValid() {
		return fmt.Errorf("conf.Set(%s): invalid key", key)
	}

	switch field.Kind() {
	case reflect.String:
		if str, ok := val.(string); ok {
			field.SetString(str)
		} else {
			return fmt.Errorf("conf.Set(%s): expected type %s, found %T", key, field.Kind().String(), val)
		}
	case reflect.Bool:
		if b, ok := val.(bool); ok {
			field.SetBool(b)
		} else {
			return fmt.Errorf("conf.Set(%s): expected type %s, found %T", key, field.Kind().String(), val)
		}
	case reflect.Int64:
		if number, ok := val.(int64); ok {
			field.SetInt(number)
		} else {
			return fmt.Errorf("conf.Set(%s): expected type %s, found %T", key, field.Kind().String(), val)
		}
	default: // type not implemented yet
		return fmt.Errorf("conf.Set(%s): type %s not implemented", key, field.Kind().String())
	}
	return nil
}

func (conf *RBAC) SetSecrets() error {
	if conf.SecretPrefix == "" { // secrets disabled?
		return nil // skip loading secrets (no aws credentials needed for local and test environments)
	}

	region, err := utils.GetCurrentRegion()

	if err != nil {
		return err
	}

	log.Println("Region:", region)
	// Create a Secrets Manager client
	svcSession, err := session.NewSession(&aws.Config{
		Region: &region,
	})
	svc := secretsmanager.New(svcSession)

	if err != nil {
		return err
	}

	err = conf.UpdateMatchingFields("{{secret:", "}}", func(field string, secretKey string) (string, error) {
		splitString := strings.Split(secretKey, ":")
		secret, err := conf.getSecretManagerSecret(svc, splitString[0], utils.IfStringSliceLength(splitString, 1))
		if err != nil {
			return "", errx.Wrap(err, "getSecretManagerSecret failed on field "+field)
		}
		return secret, nil // update field with the secret from Secrets Manager
	})
	if err != nil {
		return errx.Wrap(err, "config.SetSecrets")
	}

	return nil
}

func (conf *RBAC) getSecretManagerSecret(svc *secretsmanager.SecretsManager, secretName string, subValue string) (string, error) {

	input := &secretsmanager.GetSecretValueInput{
		SecretId:     aws.String(conf.SecretPrefix + secretName),
		VersionStage: aws.String("AWSCURRENT"),
	}

	result, err := svc.GetSecretValue(input)
	if err != nil {
		if aerr, ok := err.(awserr.Error); ok {
			switch aerr.Code() {
			case secretsmanager.ErrCodeDecryptionFailure:
				// Secrets Manager can't decrypt the protected secret text using the provided KMS key.
				return "", fmt.Errorf("%s/%s", secretsmanager.ErrCodeDecryptionFailure, err.Error())

			case secretsmanager.ErrCodeInternalServiceError:
				// An error occurred on the server side.
				return "", fmt.Errorf("%s/%s", secretsmanager.ErrCodeInternalServiceError, err.Error())

			case secretsmanager.ErrCodeInvalidParameterException:
				// You provided an invalid value for a parameter.
				return "", fmt.Errorf("%s/%s", secretsmanager.ErrCodeInvalidParameterException, err.Error())

			case secretsmanager.ErrCodeInvalidRequestException:
				// You provided a parameter value that is not valid for the current state of the resource.
				return "", fmt.Errorf("%s/%s", secretsmanager.ErrCodeInvalidRequestException, err.Error())

			case secretsmanager.ErrCodeResourceNotFoundException:
				// We can't find the resource that you asked for.
				return "", fmt.Errorf("%s/%s", secretsmanager.ErrCodeResourceNotFoundException, err.Error())
			}
		}
		return "", err
	}

	// Decrypts secret using the associated KMS CMK.
	// Depending on whether the secret is a string or binary, one of these fields will be populated.
	var resultingSecret string
	if result.SecretString != nil {
		resultingSecret = *result.SecretString
	} else {
		decodedBinarySecretBytes := make([]byte, base64.StdEncoding.DecodedLen(len(result.SecretBinary)))
		len, err := base64.StdEncoding.Decode(decodedBinarySecretBytes, result.SecretBinary)
		if err != nil {
			return "", err
		}
		resultingSecret = string(decodedBinarySecretBytes[:len])
	}

	if subValue == "" {
		return resultingSecret, nil
	}

	var jsonResult map[string]interface{}
	if err = json.Unmarshal([]byte(resultingSecret), &jsonResult); err != nil {
		return "", err
	}

	return utils.ParseUnknownToString(jsonResult[subValue])
}

type updateMatchingFieldFunc func(fieldName string, trimmed string) (string, error)

// UpdateMatchingFields iterates over string fields checking if a field value has the given prefix.
// The updFunc is called with the fieldName and the trimmed value (with no prefix and suffix).
// The returned value from updFunc is used to update the field; if there is an error then that error is returned.
func (conf *RBAC) UpdateMatchingFields(prefix, suffix string, updFunc updateMatchingFieldFunc) error {
	ptyp := reflect.TypeOf(conf)  // pointer type: *RBAC
	pval := reflect.ValueOf(conf) // pointer value: &RBAC{...}
	typ := ptyp.Elem()            // element type: RBAC
	val := pval.Elem()            // element value: RBAC{...} (can be modified because is coming from a pointer)

	num := typ.NumField()
	for i := 0; i < num; i++ {
		fieldTyp := typ.Field(i)
		fieldVal := val.Field(i)
		if fieldVal.Kind() == reflect.String {
			value := fieldVal.String()
			if tagKey, ok := trimPreffixAndSuffix(value, prefix, suffix); ok {
				updatedValue, err := updFunc(fieldTyp.Name, tagKey)
				if err != nil {
					return err // break iteration, return this error
				}
				fieldVal.SetString(updatedValue) // Update field with new value from the updFunc callback
			}
		}
	}
	return nil
}

// trimPreffixAndSuffix returns (trimmed, true) if the fieldValue is in the format preffix+trimmed+suffix.
func trimPreffixAndSuffix(fieldValue, preffix, suffix string) (string, bool) {
	if strings.HasPrefix(fieldValue, preffix) {
		trimmed := strings.TrimSuffix(strings.TrimPrefix(fieldValue, preffix), suffix)
		return trimmed, true
	}
	return fieldValue, false
}
