# End to End tests

The end to end tests seek to give us high confidence the client, api, and database work as expected. These serve as the bulk of the testing for RBAC. This is because the application relies heavily on a database and it's easier to test using a real database. Plus, with docker, it's easy to do.

To run all end-to-ends tests using local tools as necessary for speed, use the make command:

```sh
make e2e
```

To run all end-to-end tests in docker containers, use the make command:

```sh
make e2e_ci
```

### Working with E2E tests

[Docker-Compose](https://docs.docker.com/compose/) is used to spin up a database, service, and client that runs that tests. The files for that are located in across `docker-compose.yml`, `docker/` and `e2e/`.

[Testify](https://github.com/stretchr/testify) is the Go library used to reduce boilerplate when testing code. Regarding Testify, we prefer `require` over `assert` to immediately stop a test when an assertion fails. Our `e2e.Suite` has `*require.Assertions` embedded so there is no need to say `s.Require().Equal`. Instead, you say `s.Equal`

**Fakes/NOOPs**: unlike regular unit tests, mocks and fakes can not be pre-configured in E2E tests. The service is running one process and the test client in another process; E2E tests only have access to API methods. Since we can not force a service dependency to fail for a given test, we have to be a little clever. The service is started with environment `e2e`, which uses the configuration in `config/env-e2e.toml`, which uses "NOOP" configurations for clients. The "NOOP" flag is used in the `main.go` file to initialize clients to other services with fake implementations. Some of them return empty responses, but some are configured to fail if some "bad" input is passed. That way the e2e test can force a failure by passing that "bad" input, then checking that the dependency failure results in the expected API error. One example of e2e client fake is `clients/owlcli/fake.go`.

## Development/Debug mode

Run debug mode:

```bash
E2E_DEBUG=on make e2e
```

Run in coverage mode to collect test coverage collection:

```bash
make e2e_coverage
```

### Development Details

Our `make` commands have `_ci` suffix counterparts. By default, the `make` commands use local tools because docker on mac is slow. The `_ci` suffix version of commands run the same commands in docker which ensures they work on any environment with docker.

Examples

* `make e2e` uses local `go` to build test binary and run tests and uses docker to run the database and server. Best to run when working locally as it's designed to be significantly faster than `make e2e_ci`.
* `make e2e_ci` uses docker for everything. Slower than `make e2e`.

Adding the **E2E_DEBUG** environment variable, the test will additionaly:

* Write service verbose logs to `./e2e_service.log`
* Leave all containers running (so you can inspect the database)

In addition, tests can be rerun with `docker-compose run --rm e2e_test`. Specific tests can be provided as well: `docker-compose run --rm e2e_test -run CompanySuite`.
Check the `ci.sh` script to see what commands are being used. If you make changes to the service and want to restart it before the next test run, use `docker-compose kill e2e_service; docker-compose rm -f e2e_service;`. If tests fail and they involve data, it's highly recommended to examine the data in the tables. Run your queries manually against the created data to debug what is happening.

Adding the **COVERAGE** environment variable will collect test coverage. This is used in CI. It is encouraged to look at the coverage report to figure out how tested your change is. You can view coverage in your browser after running the tests with:

```bash
go tool cover -html=coverfile_e2e.out
```

To see view coverage of the files you have changed on a branch, run:

```bash
./scripts/diff_coverage.sh coverfile_e2e.out
```


### Examining the database

At the time of writing, the most valuable service logs are coming from the database layer: all queries and their args are logged. The database container is useful to poke around database tables to ensure data is there as you expect.

```sh
# Getting into the db container and psql
docker-compose exec db /bin/bash -c "psql -U rbac_admin rbac"
psql (11.1 (Debian 11.1-3.pgdg90+1))
Type "help" for help.

rbac=\#
```
