package e2e

import (
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *CompanySuite) Test_OffboardResource() {
	company := s.createCompany()
	game := s.createOnboardedGameByAdmin(company.Id)
	_ = s.createOnboardedGameByAdmin(company.Id)
	_ = s.createOnboardedGameByAdmin(company.Id)

	// actions should be: app created, game approved
	s.checkAuthedGameActionsLen(game.Id, WhitelistAdminTwitchId, company.Id, 2)

	// Verify the number of games
	gamesResp, err := s.RBAC.GetGamesByCompany(s.AdminCtx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Len(gamesResp.Games, 3)
	s.Equal(int32(3), gamesResp.XTotal)

	// Offboard
	_, err = s.RBACAdmin.OffboardResource(s.AdminCtx, &rbacrpc.OffboardResourceRequest{
		ResourceId:   fmt.Sprintf("%d", game.Id),
		ResourceType: "game",
		CompanyId:    company.Id,
	})
	s.NoError(err)

	// Verify the number of games is one less
	gamesResp, err = s.RBAC.GetGamesByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Len(gamesResp.Games, 2)
	s.Equal(int32(2), gamesResp.XTotal)

	// actions should be: app created, game approved, and game removed
	s.checkAuthedGameActionsLen(game.Id, WhitelistAdminTwitchId, company.Id, 3)
}

func (s *CompanySuite) Test_OffboardResource_RequiredArgs() {
	for _, invalidRequest := range []*rbacrpc.OffboardResourceRequest{
		{CompanyId: "asdf"},
		{CompanyId: "asdf", ResourceId: "asdf"},
		{CompanyId: "asdf", ResourceType: "asdf"},
		{ResourceId: "asdf"},
		{ResourceId: "asdf", ResourceType: "asdf"},
		{ResourceType: "asdf"},
	} {
		_, err := s.RBACAdmin.OffboardResource(s.AdminCtx, invalidRequest)
		s.EqualErrorCode(err, twirp.InvalidArgument)
	}

}

func (s *CompanySuite) Test_OffboardResource_UnknownResource() {
	company := s.createCompany()

	unknownID := unknownRandomTwitchID()
	_, err := s.RBACAdmin.OffboardResource(s.AdminCtx, &rbacrpc.OffboardResourceRequest{
		ResourceId:   unknownID,
		ResourceType: "game",
		CompanyId:    company.Id,
	})
	s.EqualErrorCode(err, twirp.NotFound)
}

func (s *CompanySuite) Test_OffboardResource_Admin() {
	company := s.createCompany()

	unknownID := unknownRandomTwitchID()
	_, err := s.RBACAdmin.OffboardResource(s.Ctx, &rbacrpc.OffboardResourceRequest{
		ResourceId:   unknownID,
		ResourceType: "game",
		CompanyId:    company.Id,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
}

func (s *CompanySuite) Test_OffboardResource_UnownedResource() {
	companyWithGame := s.createCompany()
	game := s.createOnboardedGame(companyWithGame.Id)
	companyWithoutGame := s.createCompany()

	// Try to offboard companyWithGames' game using companyWithoutGame
	_, err := s.RBACAdmin.OffboardResource(s.AdminCtx, &rbacrpc.OffboardResourceRequest{
		ResourceId:   fmt.Sprintf("%d", game.Id),
		ResourceType: "game",
		CompanyId:    companyWithoutGame.Id,
	})
	s.Error(err)

	twerr, ok := err.(twirp.Error)
	s.True(ok)
	s.Equal(twerr.Code(), twirp.NotFound)
}
