package e2e

import (
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *CompanySuite) Test_TransferCompanyOwnership() {
	company := s.createCompany()

	owner := s.membershipWithRole(company.Id, "Owner")
	admin := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
	})

	// when we expect the API to error,
	// this verifies that owner and admin were not switched
	verifyRolesNotChanged := func() {
		// assert owner is still Owner
		ownersNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
			CompanyId: company.Id,
			TwitchId:  owner.TwitchId,
		})
		s.NoError(err)
		s.Equal("Owner", ownersNewMemb.Role)

		// assert admin is still admin
		adminsNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
			CompanyId: company.Id,
			TwitchId:  admin.TwitchId,
		})
		s.NoError(err)
		s.Equal("Administrator", adminsNewMemb.Role)
	}

	// company id required
	_, err := s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         "",
		OwnersNewRole:     "Administrator",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.EqualTwirp(err, twirp.InvalidArgument, "company_id is required")
	verifyRolesNotChanged()

	// with invalid company
	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         "asdf",
		OwnersNewRole:     "Administrator",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.EqualTwirp(err, twirp.InvalidArgument, "company_id invalid UUID format")
	verifyRolesNotChanged()

	// without new role
	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.EqualTwirp(err, twirp.InvalidArgument, "owners_new_role must be one of Owner,Administrator,Developer,Marketer,Manager,Billing_Manager")
	verifyRolesNotChanged()

	// with invalid role
	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "asdf",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.EqualErrorCode(err, twirp.InvalidArgument)
	verifyRolesNotChanged()

	// cannot transfer to invalid user
	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "Administrator",
		TwitchIdToPromote: "",
	})
	s.EqualTwirp(err, twirp.InvalidArgument, "twitch_id_to_promote is required")

	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "Administrator",
		TwitchIdToPromote: randomTwitchID(),
	})
	s.EqualTwirp(err, twirp.FailedPrecondition, "twitch_id_to_promote must be in company")
	verifyRolesNotChanged()

	// requires admin context
	_, err = s.RBACAdmin.TransferCompanyOwnership(s.Ctx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "Administrator",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
	verifyRolesNotChanged()

	// happy path
	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "Administrator",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.NoError(err)

	// assert owner is Administrator
	ownersNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  owner.TwitchId,
	})
	s.NoError(err)
	s.Equal("Administrator", ownersNewMemb.Role)
	s.checkAuthedEntityActionsLen(ownersNewMemb.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 2)

	// assert admin is Owner
	adminsNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  admin.TwitchId,
	})
	s.NoError(err)
	s.Equal("Owner", adminsNewMemb.Role)
	s.checkAuthedEntityActionsLen(adminsNewMemb.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 2)

}

func (s *CompanySuite) Test_TransferCompanyOwnership_NoOwner() {
	company := s.createCompany()

	owner := s.membershipWithRole(company.Id, "Owner")
	admin := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
	})

	_, err := s.RBAC.RemoveCompanyMembership(s.AdminCtx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  owner.TwitchId,
	})
	s.NoError(err)
	s.checkAuthedEntityActionsLen(owner.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 2)

	_, err = s.RBACAdmin.TransferCompanyOwnership(s.AdminCtx, &rbacrpc.TransferCompanyOwnershipRequest{
		CompanyId:         company.Id,
		OwnersNewRole:     "",
		TwitchIdToPromote: admin.TwitchId,
	})
	s.NoError(err)

	adminsNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  admin.TwitchId,
	})
	s.NoError(err)
	s.Equal("Owner", adminsNewMemb.Role)
	s.checkAuthedEntityActionsLen(adminsNewMemb.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 2)
}
