#!/bin/bash
# For continuous integration: spin up all containers, then run all the e2e tests.

set -e

function stopContainers() {
    docker-compose down -v
}

function log() {
    echo
    echo "==> $@"
}

function logWithLocal() {
    SUFFIX="locally"
    if [ $LOCAL = "off" ]; then
        SUFFIX="in docker"
    fi

    log "$@ ${SUFFIX}"
}

function cleanup() {
    log "Cleanup"
    log "saving db logs as e2e_db.log"
    docker-compose --no-ansi logs db > ./e2e_db.log 2>&1
    if [ -z "$E2E_DEBUG" ]; then # is E2E_DEBUG not set?
        # Print logs, stop and remove containers
        time docker-compose stop e2e_service
        docker-compose kill e2e_service
        docker-compose logs e2e_service | tee e2e_service.log
        # Run this before we stop the containers
        if [ $LOCAL = "off" ]; then
            docker-compose run --rm --entrypoint "./e2e/run.sh collect" compile
        else
            ./e2e/run.sh collect
        fi
        stopContainers
    else
        log "Logs for e2e_service saved as ./e2e_service.log"
        docker-compose --no-ansi logs e2e_service > ./e2e_service.log 2>&1
        docker-compose --no-ansi logs db > ./e2e_db.log 2>&1
    fi


    cat e2e_service.log | grep "twirpErrorCode=internal" && echo "internal error was found. if these are expected errors change their type to 4xx" && exit 1

    exit $1 # exit with original error
}
trap cleanup EXIT

# Set COVERAGE to an empty string to avoid the ugly docker warning "Defaulting to a blank string."
if [ -z "$COVERAGE" ]; then
    export COVERAGE=''
fi

# Set E2E_DEBUG to an empty string to avoid the ugly docker warning "Defaulting to a blank string."
if [ -z "$E2E_DEBUG" ]; then
    export E2E_DEBUG=''
fi

LOCAL=${LOCAL:-off}

log "Cleanup containers if they are running"
stopContainers

log "Spin up DB container ..."
docker-compose up -d db

logWithLocal "Ensuring e2e tests compile"
if [ $LOCAL = "off" ]; then
    docker-compose run --rm compile
else
    # separated from compile step to natively run
    ./e2e/run.sh e2e_tests
    # this will be used in e2e_service, build for linux
    GOOS=linux ./e2e/run.sh compile
fi

# No need to run this in CI. building tests and binary is slow enough
# for the db to start up and be ready in time. when working locally,
# tests and binary may be cached and instant.
if [ $LOCAL != "off" ]; then
    log "Wait until DB is ready ..."
    # Wait until the container is ready to receive traffic by looking into the logs.
    # NOTE: this only works before the logs have shown (pipe only aborts when following new lines).
    # Using grep -q: https://unix.stackexchange.com/questions/45941/tail-f-until-text-is-seen
    # but we could have used a wait-for script instead (https://docs.docker.com/compose/startup-order/).
    # If you want to see the output for debugging the db init, use:
    # docker-compose --no-ansi logs -f db | sed '/LOG:  database system is shut down/ q'
    docker-compose --no-ansi logs -f db | grep -q 'database system is shut down'
fi

log "Spin up e2e_service ..."
docker-compose up -d e2e_service

logWithLocal "Run e2e_test ..."
if [ $LOCAL = "off" ]; then
    docker-compose run --rm e2e_test
else
    # grab the e2e_service container's host port and use that for the tests
    SERVER_ADDR=`docker inspect --format 'http://0.0.0.0:{{ index .NetworkSettings.Ports "9000/tcp" 0 "HostPort" }}' devsite-rbac_e2e_service_1` ./e2e/run.sh test
fi
