package e2e

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"sort"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"code.justin.tv/devrel/devsite-rbac/rpc/testutil"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"
)

const (
	WhitelistAdminTwitchId = "TEST_ADMIN_TWITCH_ID"
)

var (
	e2eServiceAddr = "http://e2e_service:9000" // Can be override by env var SERVER_ADDR
)

// Suite is the general suite that other e2e tests should extend.
// Before the suite starts, it initializes clients, domain and contexts.
type Suite struct {
	suite.Suite
	// Enables using assertions on the suite rather having calling s
	// everytime.
	*require.Assertions

	HTTPClient        *http.Client              // HTTP client used to configure the Twirp clients
	RBAC              rbacrpc.RBAC              // Service Twirp client
	RBACActionHistory rbacrpc.RBACActionHistory // Service Twitp client
	RBACAdmin         rbacrpc.RBACAdmin         // Service Twitp client
	Privacy           rbacrpc.Privacy

	Ctx      context.Context // requests from DevSite edge (end-users)
	AdminCtx context.Context // requests from internal sites (Vienna admins)
}

// SetupSuite is executed only once per suite, before the first test case
func (s *Suite) SetupSuite() {
	if testing.Short() {
		s.T().Skip() // prevent the test from being run with `go test ./... -short`
	}

	// Require by default: tests fail on first error instead of continue failing assertions
	s.Assertions = s.Require()

	if os.Getenv("SERVER_ADDR") != "" {
		e2eServiceAddr = os.Getenv("SERVER_ADDR")
	}

	// Setup twirp clients to make requests
	s.HTTPClient = &http.Client{Timeout: 10 * time.Second}
	s.RBAC = rbacrpc.NewRBACProtobufClient(e2eServiceAddr, s.HTTPClient)
	s.RBACActionHistory = rbacrpc.NewRBACActionHistoryProtobufClient(e2eServiceAddr, s.HTTPClient)
	s.RBACAdmin = rbacrpc.NewRBACAdminProtobufClient(e2eServiceAddr, s.HTTPClient)
	s.Privacy = rbacrpc.NewPrivacyProtobufClient(e2eServiceAddr, s.HTTPClient)

	s.InitCtx()

	// Wait until service is up (or timeout)
	s.waitUntilServiceIsReady(e2eServiceAddr)
}

// SetupTest is executed before each test
func (s *Suite) SetupTest() {
	s.InitCtx() // reset context
}

func (s *Suite) InitCtx() {
	s.Ctx = context.Background()
	s.AdminCtx = s.withAuth(context.Background(), WhitelistAdminTwitchId) // super-admin by default
}

//
// Test Helpers
//

// waitUntilServiceIsReady calls /helth until there is no error or times out.
// This allows for easy orchestration on end-to-end tests (the server container needs to be running) as well as testing our health check used by our load balancer.
func (s *Suite) waitUntilServiceIsReady(url string) {
	var err error
	attempts := 30
	waitPerAttemp := 1 * time.Second

	req, err := http.NewRequest("GET", url+"/health", nil)
	s.NoError(err, "failed creating health check request")

	var resp *http.Response
	for attempts > 0 {
		resp, err = s.HTTPClient.Do(req)
		if err == nil && resp.StatusCode == 200 {
			return // no error => the service is ready!
		}
		time.Sleep(waitPerAttemp)
		attempts -= 1
	}
	s.NoError(err, "waitUntilServiceIsReady: connection timeout. Error: %s", err.Error())
}

func (s *Suite) createCompanyApplication() string {
	companyAppRequest := newCompanyApplicationRequest()
	companyApp, err := s.RBAC.CreateCompanyApplication(s.Ctx, companyAppRequest)
	s.NoError(err)
	return companyApp.Id // ID is the only field returned by the API
}

func (s *Suite) createAndGetCompanyApplication() *rbacrpc.CompanyApplication {
	companyAppId := s.createCompanyApplication()
	companyApp, err := s.RBAC.GetCompanyApplication(s.Ctx, &rbacrpc.Id{Id: companyAppId})
	s.NoError(err)
	return companyApp
}

func (s *Suite) createCompany() *rbacrpc.Company {
	return s.onboardCompany(s.createCompanyApplication())
}

func (s *Suite) onboardCompany(companyAppId string) *rbacrpc.Company {
	onboardResp, err := s.RBAC.OnboardCompany(s.AdminCtx, &rbacrpc.OnboardCompanyRequest{
		Id: companyAppId,
	})
	s.NoError(err)
	s.NotNil(onboardResp.Company)
	s.NotEmpty(onboardResp.Company.Id)
	return onboardResp.Company
}

func (s *Suite) createGameApplicationWithCtx(ctx context.Context, companyID string) *rbacrpc.GameApplication {
	gameApp, err := s.RBAC.CreateGameApplication(ctx, &rbacrpc.CreateGameApplicationRequest{
		GameId:    randomGameID(),
		CompanyId: companyID,
	})
	s.NoError(err)
	return gameApp
}

func (s *Suite) createGameApplication(companyID string) *rbacrpc.GameApplication {
	return s.createGameApplicationWithCtx(s.Ctx, companyID)
}

// passes ctx to createGameApplicationWithCtx. enables createOnboardedGameByAdmin
func (s *Suite) createOnboardedGameWithCreateCtx(ctx context.Context, companyID string) *rbacrpc.Game {
	gameApp := s.createGameApplicationWithCtx(ctx, companyID)
	resp, err := s.RBAC.OnboardGame(s.AdminCtx, &rbacrpc.OnboardGameRequest{
		GameApplicationId: gameApp.Id,
	})
	s.NoError(err)

	return &rbacrpc.Game{
		Id:   resp.GameId,
		Name: resp.GameName,
	}
}

func (s *Suite) createOnboardedGame(companyID string) *rbacrpc.Game {
	return s.createOnboardedGameWithCreateCtx(s.Ctx, companyID)
}

func (s *Suite) createOnboardedGameByAdmin(companyID string) *rbacrpc.Game {
	return s.createOnboardedGameWithCreateCtx(s.AdminCtx, companyID)
}

func (s *Suite) createResources(company *rbacrpc.Company, resourcesToAdd int, numResourceTypes int) ([]*rbacrpc.Resource, []string) {
	resources := []*rbacrpc.Resource{}

	resourceTypes := []string{}
	for i := 0; i < numResourceTypes; i++ {
		resourceTypes = append(resourceTypes, randomString(10))
	}

	for _, resourceType := range resourceTypes {
		for i := 0; i < resourcesToAdd; i++ {
			resource := &rbacrpc.Resource{
				ExternalId: randomString(16),
				Type:       resourceType,
			}

			_, err := s.RBAC.CreateResource(s.Ctx, &rbacrpc.CreateResourceRequest{
				CompanyId: company.Id,
				Resource:  resource,
			})
			s.NoError(err)
			resources = append(resources, resource)
		}
	}

	return resources, resourceTypes
}

type membershipParams struct {
	CompanyID          string
	Role               string
	TwitchID           string // defaults to randomTwitchID()
	RequestingTwitchID string // requesting user. If empty, the request context is s.AdminCtx (whitelist-admin)
}

// Create a membership with parameters, using some defaults
func (s *Suite) createMembership(params membershipParams) (*rbacrpc.Membership, error) {
	if params.TwitchID == "" {
		if strings.ToLower(params.Role) == "billing_manager" {
			params.TwitchID = randomTwitchIDWithTIMsEnabled() // ensure that TIMs fake allows this role to be created
		} else {
			params.TwitchID = randomTwitchID()
		}
	}

	ctx := s.Ctx
	if params.RequestingTwitchID == "" {
		ctx = s.AdminCtx // whitelisted user with super-admin permissions by default
	}

	memb := &rbacrpc.Membership{
		TwitchId:  params.TwitchID,
		CompanyId: params.CompanyID,
		Role:      params.Role,

		FirstName:  randomString(10),
		LastName:   randomString(10),
		DevTitle:   randomString(10),
		CreatedAt:  fixedTimeStr(0),
		ModifiedAt: "",
	}

	_, err := s.RBAC.AddUserToCompany(ctx, &rbacrpc.AddUserToCompanyRequest{
		User: &rbacrpc.CreateUserRequest{
			TwitchId:  memb.TwitchId,
			FirstName: memb.FirstName,
			LastName:  memb.LastName,
			Title:     memb.DevTitle,
		},
		CompanyId:          memb.CompanyId,
		Role:               memb.Role,
		RequestingTwitchId: params.RequestingTwitchID,
	})
	if err != nil {
		return nil, err
	}

	return memb, nil
}

// Create a membership and assert no error
func (s *Suite) mustCreateMembership(params membershipParams) *rbacrpc.Membership {
	memb, err := s.createMembership(params)
	s.NoError(err, "Unexpected error after creating membershis with params: %+v", params)
	return memb
}

// Create a member for each available role
func (s *Suite) createMembershipsForAllRoles(companyID string) int {
	resp, err := s.RBAC.ListAllRoles(s.Ctx, &rbacrpc.Empty{})
	s.NoError(err)

	for _, role := range resp.Roles {
		if role == "Owner" {
			continue // skip because an "Owner" already exists and there can only be one on each company
		}

		s.mustCreateMembership(membershipParams{
			CompanyID: companyID,
			Role:      role,
		})
	}

	return len(resp.Roles) - 1
}

func (s *Suite) checkEntityActions(entityID, entityType, companyID string, actions []string) {
	list, err := s.RBACActionHistory.ListActionHistories(s.AdminCtx, &rbacrpc.ListActionHistoriesRequest{
		EntityId:   entityID,
		EntityType: entityType,
		Limit:      uint64(len(actions)),
	})
	s.NoError(err, "%s: %s", entityType, entityID)
	s.Equal(uint64(len(actions)), list.Total)

	loadedActions := []string{}
	for _, history := range list.ActionHistories {
		loadedActions = append(loadedActions, history.Action)
	}
	// These can sometimes come back out of order due their timestamps
	// going down to second granularity.
	sort.Strings(actions)
	sort.Strings(loadedActions)
	s.Equal(actions, loadedActions)
}

func (s *Suite) checkGameActions(entityID int32, companyID string, actions []string) {
	s.checkEntityActions(fmt.Sprintf("%d", entityID), "Game", companyID, actions)
}

func (s *Suite) checkAuthedGameActionsLen(entityID int32, userTwitchID, companyID string, length int) {
	s.checkAuthedEntityActionsLen(fmt.Sprintf("%d", entityID), "Game", userTwitchID, companyID, length)
}

// check history logs length and ensure the userTwitchID is set to the same value
func (s *Suite) checkAuthedEntityActionsLen(entityID, entityType, userTwitchID, companyID string, length int) {
	list, err := s.RBACActionHistory.ListActionHistories(s.AdminCtx, &rbacrpc.ListActionHistoriesRequest{
		EntityId:   entityID,
		EntityType: entityType,
		CompanyId:  companyID,
		Limit:      100,
	})
	s.NoError(err, "%s: %s", entityType, entityID)
	s.Len(list.ActionHistories, length, "%s: %s", entityType, entityID)

	for _, history := range list.ActionHistories {
		s.Equal(userTwitchID, history.UserTwitchId, fmt.Sprintf("%v", history))
		s.Equal(companyID, history.CompanyId, fmt.Sprintf("%v", history))
		s.NotEmpty(history.CreatedAt, fmt.Sprintf("%v", history))
		s.NotEmpty(history.EntityId, fmt.Sprintf("%v", history))
		s.NotEmpty(history.EntityType, fmt.Sprintf("%v", history))
		s.NotEmpty(history.Action, fmt.Sprintf("%v", history))
	}
}

func (s *Suite) checkAuthedCompanyActionsLen(companyID, userTwitchID string, length int) {
	list, err := s.RBACActionHistory.ListActionHistories(s.AdminCtx, &rbacrpc.ListActionHistoriesRequest{
		CompanyId: companyID,
		Limit:     100,
	})
	s.NoError(err)
	s.Len(list.ActionHistories, length)

	for _, history := range list.ActionHistories {
		s.Equal(userTwitchID, history.UserTwitchId, fmt.Sprintf("%v", history))
		s.Equal(companyID, history.CompanyId, fmt.Sprintf("%v", history))
		s.NotEmpty(history.CreatedAt, fmt.Sprintf("%v", history))
		s.NotEmpty(history.EntityId, fmt.Sprintf("%v", history))
		s.NotEmpty(history.EntityType, fmt.Sprintf("%v", history))
		s.NotEmpty(history.Action, fmt.Sprintf("%v", history))
	}
}

type permissionCase struct {
	Role string
	Pass []*rbacrpc.ValidateQuery
	Fail []*rbacrpc.ValidateQuery
}

type permissionOpts struct {
	SkipCreatingMemberships bool
}

// randomRole is a fake role that tests a user not in the company.
const randomRole = "Random"

func (s *Suite) checkCompanyRolePermissions(companyID string, testCases []permissionCase, opts *permissionOpts) {
	checkValidate := func(name string, user *rbacrpc.Membership, pass, fail []*rbacrpc.ValidateQuery) {
		msg := "%s: %v"

		s.True(len(pass) != 0 || len(fail) != 0, msg, name, "expected either pass or fail array to have length > 0")

		for _, p := range pass {
			p.UserId = user.TwitchId
			resp, err := s.RBAC.ValidateByTwitchID(s.Ctx, p)
			s.NoError(err, msg, name, p)
			s.True(resp.Valid, msg, name, p)
		}

		for _, f := range fail {
			f.UserId = user.TwitchId
			resp, err := s.RBAC.ValidateByTwitchID(s.Ctx, f)
			s.NoError(err, msg, name, f)
			s.False(resp.Valid, msg, name, f)
		}
	}

	// create all roles
	if opts == nil || !opts.SkipCreatingMemberships {
		s.createMembershipsForAllRoles(companyID)
	}

	// validate all roles in company were validated
	// ensures that new roles have permissions tested
	foundRoles := map[string]bool{
		randomRole: false,
	}
	rolesResp, err := s.RBAC.ListAllRoles(s.Ctx, &rbacrpc.Empty{})
	s.NoError(err)
	for _, role := range rolesResp.Roles {
		foundRoles[role] = false
	}

	s.Equal(len(foundRoles), len(testCases), "are all company roles being tested?")

	// validate provided permissions
	for _, testCase := range testCases {
		foundRoles[testCase.Role] = true
		user := &rbacrpc.Membership{TwitchId: randomTwitchID()}
		if testCase.Role != randomRole {
			user = s.membershipWithRole(companyID, testCase.Role)
		}

		checkValidate(testCase.Role, user, testCase.Pass, testCase.Fail)
	}

	for role, found := range foundRoles {
		s.True(found, "role %q was not tested", role)
	}
}

func (s *Suite) membershipsWithRole(companyID string, role string) []*rbacrpc.Membership {
	resp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{
		Id:    companyID,
		Role:  role,
		Limit: 100,
	})
	s.NoError(err)
	return resp.Memberships
}

func (s *Suite) membershipWithRole(companyID string, role string) *rbacrpc.Membership {
	resp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{
		Id:    companyID,
		Role:  role,
		Limit: 1,
	})
	s.NoError(err)
	s.Len(resp.Memberships, 1, "Expected one member with role "+role)
	s.NotNil(resp.Memberships[0])
	return resp.Memberships[0]
}

func (s *Suite) membershipRoleCounts(companyID string) membershipCounts {
	resp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{
		Id:    companyID,
		Limit: 100,
	})
	s.NoError(err)
	return countMemberships(resp.Memberships)
}

type membershipCounts struct {
	Owners     int
	Admins     int
	Managers   int
	Marketers  int
	Developers int
}

// counts memberships by role and returns a struct with typed role fields.
func countMemberships(memberships []*rbacrpc.Membership) membershipCounts {
	var m membershipCounts
	for _, u := range memberships {
		// do a bit of extra work here to map role strings to
		// fields so role counts are easier to access in tests
		switch u.Role {
		case "Owner":
			m.Owners += 1
		case "Administrator":
			m.Admins += 1
		case "Manager":
			m.Managers += 1
		case "Marketer":
			m.Marketers += 1
		case "Developer":
			m.Developers += 1
		}
	}

	return m
}

// EqualErrorCode is a convenience method for testutil.EqualErrorCode
func (s *Suite) EqualErrorCode(err error, code twirp.ErrorCode, msgAndArgs ...interface{}) {
	s.Error(err, msgAndArgs...)
	testutil.EqualErrorCode(s.T(), code, err, msgAndArgs...)
}

// EqualErrorMsg is a convenience method for testutil.EqualErrorMsg
func (s *Suite) EqualErrorMsg(err error, msg string, msgAndArgs ...interface{}) {
	s.Error(err, msgAndArgs...)
	testutil.EqualErrorMsg(s.T(), msg, err, msgAndArgs...)
}

// EqualErrorMeta is a convenience method to check a Twirp error metadata value
func (s *Suite) EqualErrorMeta(err error, metaKey, metaValue string, msgAndArgs ...interface{}) {
	s.Error(err, msgAndArgs...)
	testutil.EqualErrorMeta(s.T(), err, metaKey, metaValue, msgAndArgs...)
}

// EqualTwirp is a convenience method for testutil.EqualErrorCode and testutil.EqualErrorMsg
func (s *Suite) EqualTwirp(err error, code twirp.ErrorCode, msg string, msgAndArgs ...interface{}) {
	s.Error(err)
	testutil.EqualErrorCode(s.T(), code, err, msgAndArgs...)
	testutil.EqualErrorMsg(s.T(), msg, err, msgAndArgs...)
}

// withAuth adds the header "Authorization: OAuth <calledId>".
// This works with the fake OAuth client in the service, that is set to authorize that callerId.
func (s *Suite) withAuth(ctx context.Context, callerId string) context.Context {
	ctx, _ = twirp.WithHTTPRequestHeaders(ctx, http.Header{
		"Authorization": []string{"OAuth " + callerId},
	})
	return ctx
}

func newCompanyApplicationRequest() *rbacrpc.CreateCompanyApplicationRequest {
	return &rbacrpc.CreateCompanyApplicationRequest{
		TwitchId:         randomTwitchID(),
		CompanyName:      "Company-" + randomString(15),
		CompanyWebsite:   "http://twitch.tv",
		CompanyType:      1,
		Industry:         "games",
		CompanySize:      "1",
		City:             "san francisco",
		State:            "ca",
		Country:          "usa",
		ContactFirstName: randomString(10),
		ContactLastName:  randomString(10),
		ContactTitle:     randomString(10),
		ProductInterest:  randomString(10),
		JoinReason:       randomString(10),
	}
}

type companyInviteParams struct {
	CompanyID       string
	Role            string
	InviteeTwitchID string // defaults to randomTwitchID()
	InviterTwitchID string // requesting user. If empty, the request context is s.AdminCtx (whitelist-admin)
}

// Create a membership and assert no error
func (s *Suite) mustCreateCompanyInvite(params companyInviteParams) *rbacrpc.CompanyInvite {
	memb, err := s.createCompanyInvite(params)
	s.NoError(err, "Unexpected error after creating company invites with params: %+v", params)
	return memb
}

// Create a company invite with parameters, using some defaults
func (s *Suite) createCompanyInvite(params companyInviteParams) (*rbacrpc.CompanyInvite, error) {
	if params.InviteeTwitchID == "" {
		if strings.ToLower(params.Role) == "billing_manager" {
			params.InviteeTwitchID = randomTwitchIDWithTIMsEnabled() // ensure that TIMs fake allows this role to be created
		} else {
			params.InviteeTwitchID = randomTwitchID()
		}
	}

	ctx := s.Ctx
	if params.InviterTwitchID == "" {
		ctx = s.AdminCtx // whitelisted user with super-admin permissions by default
	}

	compInv, err := s.RBAC.CreateCompanyInvite(ctx, &rbacrpc.CreateCompanyInviteRequest{
		InviteeTwitchId: params.InviteeTwitchID,
		InviterTwitchId: params.InviterTwitchID,
		CompanyId:       params.CompanyID,
		Role:            params.Role,
	})
	if err != nil {
		return nil, err
	}

	return compInv, nil
}
