package e2e

import (
	"testing"

	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/clients/owlcli"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func Test_ExtensionSuite(t *testing.T) {
	s := &ExtensionSuite{}
	suite.Run(t, s)
}

type ExtensionSuite struct {
	Suite
}

func (s *ExtensionSuite) Test_TransferExtension_And_Detach_Success() {
	extensionID := "client-id-00001"
	validOwnerID := owlcli.FakeOwnerIDForClientID(extensionID)
	company, err := s.createCompanyAndTransferExtension(validOwnerID, "", extensionID)
	s.NoError(err)
	s.True(s.hasCompanyExtension(company.Id, extensionID))
	_, err = s.RBAC.DetachExtensionFromCompany(s.AdminCtx, &rbacrpc.DetachExtensionRequest{
		CompanyId:   company.Id,
		ExtensionId: extensionID,
	})
	s.NoError(err)
	s.False(s.hasCompanyExtension(company.Id, extensionID))
}

func (s *ExtensionSuite) Test_TransferExtension_And_Detach_Success_Monetized() {
	extensionID := "client-id-00002"
	validOwnerID := owlcli.FakeOwnerIDForClientID(extensionID)
	company, err := s.createCompanyAndTransferExtension(validOwnerID, randomTwitchIDWithTIMsEnabled(), extensionID)
	s.NoError(err)
	s.True(s.hasCompanyExtension(company.Id, extensionID))
	_, err = s.RBAC.DetachExtensionFromCompany(s.AdminCtx, &rbacrpc.DetachExtensionRequest{
		CompanyId:   company.Id,
		ExtensionId: extensionID,
		OwnerId:     randomTwitchIDWithTIMsEnabled(),
	})
	s.NoError(err)
	s.False(s.hasCompanyExtension(company.Id, extensionID))
}

func (s *ExtensionSuite) Test_TransferExtension_InvalidExtensionId() {
	extensionID := "invalid-asdfgghhfghf" // "invalid" triggers fake error
	company, err := s.createCompanyAndTransferExtension(randomTwitchID(), "", extensionID)
	s.EqualError(err, "twirp error invalid_argument: extension_id is not a valid client id")
	s.False(s.hasCompanyExtension(company.Id, extensionID))
}

func (s *ExtensionSuite) Test_TransferExtension_OwnerNotPartOfOrg() {
	extensionID := randomTwitchID()
	company, err := s.createCompanyAndTransferExtension("0", "", extensionID)
	s.EqualError(err, "twirp error not_found: user is not a member of the company")
	s.False(s.hasCompanyExtension(company.Id, extensionID))
}

func (s *ExtensionSuite) Test_DetachExtension_ClientNotFound() {
	extensionID := "client-id-00003"
	validOwnerID := owlcli.FakeOwnerIDForClientID(extensionID)
	company, err := s.createCompanyAndTransferExtension(validOwnerID, "", extensionID)
	s.NoError(err)
	_, err = s.RBAC.DetachExtensionFromCompany(s.AdminCtx, &rbacrpc.DetachExtensionRequest{
		CompanyId: company.Id,
	})
	s.EqualErrorCode(err, twirp.InvalidArgument)
	s.True(s.hasCompanyExtension(company.Id, extensionID))
}

func (s *ExtensionSuite) Test_DetachExtension_InvalidOwnerId() {
	extensionID := "client-id-00004"
	validOwnerID := owlcli.FakeOwnerIDForClientID(extensionID)
	company, err := s.createCompanyAndTransferExtension(validOwnerID, randomTwitchIDWithTIMsEnabled(), extensionID)
	s.NoError(err)
	_, err = s.RBAC.DetachExtensionFromCompany(s.AdminCtx, &rbacrpc.DetachExtensionRequest{
		CompanyId:   company.Id,
		ExtensionId: extensionID,
		OwnerId:     "-1", // invalid owner id
	})
	s.EqualError(err, "twirp error failed_precondition: TIMS required for Billing Manager role")
	s.True(s.hasCompanyExtension(company.Id, extensionID))
}

//
// Test Helpers
//

func (s *ExtensionSuite) createCompanyAndTransferExtension(ownerID, newOwnerID, clientID string) (*rbacrpc.Company, error) {
	company := s.createCompany()

	s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
		TwitchID:  ownerID, // must be the owlcli.FakeOwnerIDForClientID(clientID) so GetClient check works
	})

	if newOwnerID != "" {
		s.mustCreateMembership(membershipParams{
			CompanyID: company.Id,
			Role:      "Billing_Manager",
			TwitchID:  newOwnerID,
		})
	}
	_, err := s.RBAC.TransferExtensionToCompany(s.AdminCtx, &rbacrpc.TransferExtensionRequest{
		CompanyId:        company.Id,
		ExtensionId:      clientID,
		BillingManagerId: newOwnerID,
	})
	return company, err
}

func (s *ExtensionSuite) hasCompanyExtension(companyID, extensionID string) bool {
	// ensure company doesn't have extension
	result, err := s.RBAC.ListResources(s.Ctx, &rbacrpc.ListResourcesRequest{
		CompanyId:    companyID,
		ResourceType: "extension",
	})
	s.NoError(err)

	found := false
	for _, resource := range result.Resources {
		if resource.ExternalId == extensionID {
			found = true
			break
		}
	}
	return found
}
