package e2e

import (
	"fmt"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
)

func Test_GameApplicationsSuite(t *testing.T) {
	s := &GameApplicationsSuite{}
	suite.Run(t, s)
}

type GameApplicationsSuite struct {
	Suite
}

func (s *GameApplicationsSuite) Test_CreateGameApplication() {
	company := s.createCompany()
	s.createGameApplication(company.Id)

	// Get and verify game applications
	appsResp, err := s.RBAC.GetGameApplicationsByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)

	gameApps := appsResp.GameApplications
	s.Equal(1, len(gameApps))
	s.Equal(1, int(appsResp.GetXTotal()))

	// No games created (game applications were not approved yet)
	gamesResp, err := s.RBAC.GetGamesByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Equal(len(gamesResp.Games), 0, "expected 0 games upon company creation")
	s.Equal(len(gamesResp.GetGames()), int(gamesResp.GetXTotal()))

	// ActivityLogsCreated
	for _, gameApp := range gameApps {
		gameIDStr := fmt.Sprintf("%d", gameApp.GameId)
		s.checkAuthedEntityActionsLen(gameIDStr, "Game", "", company.Id, 1)
	}
}

func (s *GameApplicationsSuite) Test_CreateGameApplication_Duplicated() {
	gameID := randomGameID()
	company := s.createCompany()

	firstGameApp, err := s.RBAC.CreateGameApplication(s.Ctx, &rbacrpc.CreateGameApplicationRequest{
		GameId:    gameID,
		CompanyId: company.Id,
	})
	s.NoError(err)
	s.NotEmpty(firstGameApp.Id)

	listResp, err := s.RBAC.ListGameApplications(s.AdminCtx, &rbacrpc.ListGameApplicationsRequest{
		Limit:     10,
		GameId:    gameID,
		CompanyId: company.Id,
	})
	s.NoError(err)
	s.Len(listResp.GameApplications, 1)

	secondGameApp, err := s.RBAC.CreateGameApplication(s.Ctx, &rbacrpc.CreateGameApplicationRequest{
		GameId:    gameID,
		CompanyId: company.Id,
	})
	s.NoError(err)
	s.Equal(firstGameApp.Id, secondGameApp.Id)

	listResp, err = s.RBAC.ListGameApplications(s.AdminCtx, &rbacrpc.ListGameApplicationsRequest{
		Limit:     10,
		GameId:    gameID,
		CompanyId: company.Id,
	})
	s.NoError(err)
	s.Len(listResp.GameApplications, 1)
}

func (s *GameApplicationsSuite) Test_CompanyApplicationWithGames_CreatesGameApplications() {
	gameID1 := fmt.Sprintf("%d", randomGameID())
	gameID2 := fmt.Sprintf("%d", randomGameID())

	// Company Application with 2 games
	companyAppRequest := newCompanyApplicationRequest()
	companyAppRequest.Games = []string{gameID1, gameID2}
	companyApp, err := s.RBAC.CreateCompanyApplication(s.Ctx, companyAppRequest)
	s.NoError(err)

	// history logs have no activity yet, game applications are created *after* the company is onboarded.
	s.checkAuthedEntityActionsLen(gameID1, "Game", WhitelistAdminTwitchId, companyApp.Id, 0)
	s.checkAuthedEntityActionsLen(gameID2, "Game", WhitelistAdminTwitchId, companyApp.Id, 0)

	// Onboard Company Application => will create game applications
	onboardCompanyResp, err := s.RBAC.OnboardCompany(s.AdminCtx, &rbacrpc.OnboardCompanyRequest{Id: companyApp.Id})
	s.NoError(err)
	company := onboardCompanyResp.Company

	// Check Game Applications created
	gameAppsResp, err := s.RBAC.ListGameApplications(s.AdminCtx, &rbacrpc.ListGameApplicationsRequest{
		CompanyId: company.Id,
	})
	s.NoError(err)
	s.Len(gameAppsResp.GameApplications, 2)
	s.Equal(int32(2), gameAppsResp.XTotal)

	// Check history logs: application was created
	actions := []string{
		"Game application created",
	}
	s.checkEntityActions(gameID1, "Game", company.Id, actions)
	s.checkEntityActions(gameID2, "Game", company.Id, actions)

	// Onboard one of those games
	gameApp := gameAppsResp.GameApplications[0]
	onboardGameResp, err := s.RBAC.OnboardGame(s.AdminCtx, &rbacrpc.OnboardGameRequest{
		GameApplicationId: gameApp.Id,
	})
	s.NoError(err)
	gameID := onboardGameResp.GameId

	// Check history logs for the onboarded game: application created and approved
	s.checkEntityActions(gameID1, "Game", company.Id, []string{
		"Game application created",
		"Game approved",
	})

	// Check game resource created
	resourcesResp, err := s.RBAC.ListResources(s.Ctx, &rbacrpc.ListResourcesRequest{
		CompanyId:    company.Id,
		ResourceType: "game",
	})
	s.NoError(err)
	s.Equal(uint64(1), resourcesResp.XTotal)
	res := resourcesResp.Resources[0]
	s.Equal(fmt.Sprintf("%d", gameID), res.ExternalId)
}

func (s *GameApplicationsSuite) Test_CompanyApplicationWithNoGames_ThenAddGames() {
	company := s.createCompany()
	game1 := s.createOnboardedGame(company.Id)
	game2 := s.createOnboardedGame(company.Id)

	// verify history logs
	actions := []string{
		"Game application created",
		"Game approved",
	}
	s.checkGameActions(game1.Id, company.Id, actions)
	s.checkGameActions(game2.Id, company.Id, actions)

	// verify games are available by user id
	admin := s.membershipWithRole(company.Id, "Owner")
	userGamesResp, err := s.RBAC.GetGamesByTwitchID(s.Ctx, &rbacrpc.Id{Id: admin.TwitchId})
	s.NoError(err)
	s.Len(userGamesResp.Games, 2)
}

func (s *GameApplicationsSuite) Test_CompanyApplicationWithNoGames_ThenAddGames_Multiorg() {
	company1 := s.createCompany()
	game1 := s.createOnboardedGame(company1.Id)

	company2 := s.createCompany()
	game2 := s.createOnboardedGame(company2.Id)

	// verify history logs
	actions := []string{
		"Game application created",
		"Game approved",
	}
	s.checkGameActions(game1.Id, company1.Id, actions)
	s.checkGameActions(game2.Id, company2.Id, actions)

	// verify games are available by user id
	admin1 := s.membershipWithRole(company1.Id, "Owner")
	admin2 := s.membershipWithRole(company2.Id, "Owner")

	_, err2 := s.RBAC.AddUserToCompany(s.AdminCtx, &rbacrpc.AddUserToCompanyRequest{
		User: &rbacrpc.CreateUserRequest{
			TwitchId:  admin1.TwitchId,
			FirstName: admin1.FirstName,
			LastName:  admin1.LastName,
			Title:     admin1.DevTitle,
		},
		CompanyId:          company2.Id,
		Role:               "Developer",
		RequestingTwitchId: admin2.TwitchId,
	})

	userGamesResp, err1 := s.RBAC.GetGamesByTwitchID(s.Ctx, &rbacrpc.Id{Id: admin1.TwitchId})

	s.NoError(err1)
	s.NoError(err2)
	s.Len(userGamesResp.Games, 2)
}

func (s *GameApplicationsSuite) Test_OnboardGame() {
	company := s.createCompany()
	game1App := s.createGameApplication(company.Id)
	game2App := s.createGameApplication(company.Id)

	// Approve games
	gameResp1, err := s.RBAC.OnboardGame(s.AdminCtx, &rbacrpc.OnboardGameRequest{GameApplicationId: game1App.Id})
	s.NoError(err)
	gameResp2, err := s.RBAC.OnboardGame(s.AdminCtx, &rbacrpc.OnboardGameRequest{GameApplicationId: game2App.Id})
	s.NoError(err)

	// verify history logs
	actions := []string{
		"Game application created",
		"Game approved",
	}
	s.checkGameActions(gameResp1.GameId, company.Id, actions)
	s.checkGameActions(gameResp2.GameId, company.Id, actions)

	// verify games are available by company
	gamesResp, err := s.RBAC.GetGamesByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Len(gamesResp.Games, 2)

	// verify games are available by user
	admin := s.membershipWithRole(company.Id, "Owner")
	userGamesResp, err := s.RBAC.GetGamesByTwitchID(s.Ctx, &rbacrpc.Id{Id: admin.TwitchId})
	s.NoError(err)
	s.Len(userGamesResp.Games, 2)
}

func (s *GameApplicationsSuite) Test_CreateCompany_DeleteGameApplications() {
	company := s.createCompany()
	_ = s.createGameApplication(company.Id)
	_ = s.createGameApplication(company.Id)

	// Get game applications
	resp, err := s.RBAC.GetGameApplicationsByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Equal(int(resp.XTotal), 2)

	gameApp := resp.GameApplications[0]
	_, err = s.RBAC.DeleteGameApplication(s.Ctx, &rbacrpc.DeleteGameApplicationRequest{
		Id: gameApp.Id,
	})
	s.NoError(err)

	resp, err = s.RBAC.GetGameApplicationsByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Equal(int(resp.XTotal), 1)

	s.checkAuthedEntityActionsLen(gameApp.GameName, "Game", "", company.Id, 2)
}
