package e2e

import (
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
)

type GetUsersByCompanyTestSuite struct {
	Suite
	testCompanyId string
}

func Test_GetUsersByCompanyTestSuite(t *testing.T) {
	s := &GetUsersByCompanyTestSuite{}
	suite.Run(t, s)
}

// SetupSuite is executed only once, before the suite runs the first test
func (s *GetUsersByCompanyTestSuite) SetupSuite() {
	s.Suite.SetupSuite()

	// since tests in this file are all read-only, we can initialize data here
	company := s.createCompany()
	s.testCompanyId = company.Id

	s.createMembershipsForAllRoles(company.Id)
}

func (s *GetUsersByCompanyTestSuite) Test_GetUsersByCompanyId() {
	resp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: s.testCompanyId})
	s.NoError(err)
	s.Len(resp.Memberships, 6)

	counts := countMemberships(resp.Memberships)
	s.Equal(uint64(6), resp.Total)
	s.Equal(1, counts.Owners)
	s.Equal(1, counts.Admins)
	s.Equal(1, counts.Managers)
	s.Equal(1, counts.Marketers)
	s.Equal(1, counts.Developers)
}

func (s *GetUsersByCompanyTestSuite) Test_GetUsersByCompanyId_WithLimitAndSort() {
	resp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{
		Id:     s.testCompanyId,
		Limit:  2,
		Offset: 2,
		SortBy: "role",
	})
	s.NoError(err)
	s.Len(resp.Memberships, 2)

	counts := countMemberships(resp.Memberships)
	s.Equal(uint64(6), resp.Total)
	s.Equal(uint64(2), resp.Offset)
	s.Equal(0, counts.Admins)
	s.Equal("Developer", resp.Memberships[0].Role)
	s.Equal("Manager", resp.Memberships[1].Role)
}

func (s *GetUsersByCompanyTestSuite) Test_GetUsersByCompanyId_WithValidRole() {
	resp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{
		Id:     s.testCompanyId,
		SortBy: "role",
		Role:   "administrator",
	})
	s.NoError(err)
	s.Len(resp.Memberships, 1)
	s.Equal(uint64(1), resp.Total)

	counts := countMemberships(resp.Memberships)
	s.Equal(0, counts.Owners)
	s.Equal(1, counts.Admins)
	s.Equal(0, counts.Managers)
	s.Equal(0, counts.Marketers)
	s.Equal(0, counts.Developers)
}

func (s *GetUsersByCompanyTestSuite) Test_GetUsersByCompanyId_WithInvalidRole() {
	_, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{
		Id:     s.testCompanyId,
		SortBy: "role",
		Role:   "asdf",
	})
	s.Error(err)
}
