package e2e

import (
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"
)

type SetExtensionBillingManagerTestSuite struct {
	Suite
}

func Test_SetExtensionBillingManagerTestSuite(t *testing.T) {
	s := &SetExtensionBillingManagerTestSuite{}
	suite.Run(t, s)
}

func (s *SetExtensionBillingManagerTestSuite) Test_SetExtensionBillingManager() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")
	administrator := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
		TwitchID:  randomTwitchID(),
	})
	billingManager := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Billing_Manager",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})
	developer := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
		TwitchID:  randomTwitchID(),
	})

	// Add extension resource
	extensionID := "valid-client-id-6" // Client-ID assumed to be valid by the fake Owl client
	_, err := s.RBAC.CreateResource(s.AdminCtx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource: &rbacrpc.Resource{
			ExternalId: extensionID,
			Type:       "extension",
		},
	})
	s.NoError(err)

	// Viena-whitelisted admin (s.AdminCtx) is able to set an extension billing manager
	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: billingManager.TwitchId,
		ExtensionClientId:      extensionID,
	})
	s.NoError(err)

	// Owner can set an extension billing manager
	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: billingManager.TwitchId,
		ExtensionClientId:      extensionID,
		RequestingTwitchId:     owner.TwitchId,
	})
	s.NoError(err)

	// Administrator can not set an extension billing manager
	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: billingManager.TwitchId,
		ExtensionClientId:      extensionID,
		RequestingTwitchId:     administrator.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Trying to set members in different companies returns PermissionDenied
	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: randomTwitchID(),
		ExtensionClientId:      extensionID,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Can not set members with non-eligible roles (e.g. Developer)
	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: developer.TwitchId,
		ExtensionClientId:      extensionID,
		RequestingTwitchId:     owner.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)
	s.EqualErrorMeta(err, "precondition", "invalid_role")

	// Extension must be apart of the organization
	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: billingManager.TwitchId,
		ExtensionClientId:      "valid-client-id-8", // valid for fake Owl client, but not an org resource
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)
	s.EqualErrorMeta(err, "precondition", "not_in_organization")

	// Selected billing manager can be the company owner, but must have TIMs enabled
	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: owner.TwitchId,
		ExtensionClientId:      extensionID,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)
	s.EqualErrorMeta(err, "precondition", "tims_required")

	// Selected billing manager can be an Administrator, but must have TIMs enabled
	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: administrator.TwitchId,
		ExtensionClientId:      extensionID,
		RequestingTwitchId:     owner.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)
	s.EqualErrorMeta(err, "precondition", "tims_required")

	administratorWithTIMs := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})
	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: administratorWithTIMs.TwitchId,
		ExtensionClientId:      extensionID,
		RequestingTwitchId:     owner.TwitchId,
	})
	s.NoError(err)
}
