package e2e

import (
	"fmt"
	"math/rand"
	"time"
)

const (
	letterBytes   = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
	letterIdxBits = 6                    // 6 bits to represent a letter index
	letterIdxMask = 1<<letterIdxBits - 1 // All 1-bits, as many as letterIdxBits
	letterIdxMax  = 63 / letterIdxBits   // # of letter indices fitting in 63 bits

	lowerIDRange             = 1
	upperIDRange             = 10000000
	BillingManagerLowerRange = upperIDRange
	BillingManagerUpperRange = upperIDRange * 3 / 2
	upperUnusedRange         = 2147483647
)

var src = rand.New(rand.NewSource(time.Now().UnixNano()))

func randomString(n int) string {
	b := make([]byte, n)
	// A rand.Int63() generates 63 random bits, enough for letterIdxMax letters!
	for i, cache, remain := n-1, src.Int63(), letterIdxMax; i >= 0; {
		if remain == 0 {
			cache, remain = src.Int63(), letterIdxMax
		}
		if idx := int(cache & letterIdxMask); idx < len(letterBytes) {
			b[i] = letterBytes[idx]
			i--
		}
		cache >>= letterIdxBits
		remain--
	}

	return string(b)
}

func randomNumberInRange(low, hi int) int {
	return low + src.Intn(hi-low)
}

func randomTwitchID() string {
	return fmt.Sprintf("%d", randomNumberInRange(lowerIDRange, upperIDRange))
}

// twitch ID that is guaranteed to not be used before. it returns a number
// from a range that is not used by randomTwitchID().
func unknownRandomTwitchID() string {
	return fmt.Sprintf("%d", randomNumberInRange(BillingManagerUpperRange, upperUnusedRange))
}

// randomTwitchIDWithTIMsEnabled returns a twitchID within a range that will not conflict randomTwitchID.
// The fake TIMS client returns valid on this twitch ID range.
func randomTwitchIDWithTIMsEnabled() string {
	return fmt.Sprintf("%d", randomNumberInRange(BillingManagerLowerRange, BillingManagerUpperRange))
}

func randomGameID() int32 {
	return int32(randomNumberInRange(lowerIDRange, upperIDRange))
}

// game ID that is outside of the range used by randomGameID.
func unknownRandomGameID() int32 {
	return int32(randomNumberInRange(upperIDRange, upperUnusedRange))
}

// fixedTimeStr returns a RFC3339 timestamp.
// The seconds arg is a number from 0 to 60, can be used to differentiate between different timestamps in tests.
func fixedTimeStr(seconds int) string {
	return fmt.Sprintf("2006-01-02 15:%02d", seconds)
}
