// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as cdk from '@aws-cdk/core';
import { DEV_ENV, PROD_ENV } from './consts';
import { VpcStack } from './vpc-stack';
import { RbacStack } from './rbac-stack';
import { DeployStack } from './deploy-stack';
import { RoutesStack } from './routes-stack';
import { CertStack } from './cert-stack';
import { PdmsCronStack } from './pdms-cron';
import { CommonStack } from './common-stack';

enum CreateStacksTarget {
  Dev = 1,
  Prod,
  Disaster,
}

function createStacks(scope: cdk.App, env: cdk.Environment, target: CreateStacksTarget) {
  const prefix = CreateStacksTarget[target];

  const vpcStack = new VpcStack(scope, prefix + 'Vpc', { env });
  const certStack = new CertStack(scope, prefix + 'Cert', { env });
  const commonStack = new CommonStack(scope, prefix + 'Common', { env });

  const rbacStack = new RbacStack(scope, prefix + 'Rbac', {
    env,
    vpc: vpcStack.vpc,
    generalVpcESecurityGroup: vpcStack.generalVpcESecurityGroup,
    vpcStack,
    rbacCert: certStack.RBAC_CERT,
  });
  rbacStack.addDependency(commonStack);

  new DeployStack(scope, prefix + 'RbacDeploy', {
    env,
    vpc: vpcStack.vpc,
    deployService: rbacStack.RBAC_SERVICE,
    githubSecurityGroup: vpcStack.githubSecurityGroup,
  });

  new RoutesStack(scope, prefix + 'RbacRoutes', {
    env,
    vpc: vpcStack.vpc,
    rbacPublicLB: rbacStack.RBAC_PUBLIC_LB,
  }).addDartZone(vpcStack);

  new PdmsCronStack(scope, prefix + 'PdmsCron', {
    env,
    vpc: vpcStack.vpc,
    rbacStack,
    commonStack,
  });
}

const app = new cdk.App();
createStacks(app, DEV_ENV, CreateStacksTarget.Dev);
createStacks(app, PROD_ENV, CreateStacksTarget.Prod);
