import { DEV_PDMS_ROLE, PROD_ACCOUNT_ID, PROD_PDMS_ROLE } from './consts';
import { StackProps, Stack, Construct, Tag, Duration } from '@aws-cdk/core';
import { SecurityGroup, IVpc, Peer, InstanceType, Port, SubnetType } from '@aws-cdk/aws-ec2';
import { Certificate } from '@aws-cdk/aws-certificatemanager';
import { Ec2Service, Cluster, ContainerImage, AwsLogDriver, Ec2TaskDefinition } from '@aws-cdk/aws-ecs';
import { ApplicationLoadBalancer, ListenerCertificate } from '@aws-cdk/aws-elasticloadbalancingv2';
import { PolicyStatement, Effect, ManagedPolicy} from '@aws-cdk/aws-iam';
import * as path from 'path';
import { VpcStack } from './vpc-stack';

interface RbacStackProps extends StackProps {
  vpc: IVpc;
  generalVpcESecurityGroup: SecurityGroup;
  vpcStack: VpcStack;
  rbacCert: Certificate;
}

/*
 *
 * When deploying this stack you will need to deploy only this stack and then update consts.ts ECR's value
 * as well as adding its IAM Role to S2S. You can find this information in the logs of the failing to start tasks.
 *
 */
export class RbacStack extends Stack {
  public RBAC_SERVICE: Ec2Service;

  public RBAC_PUBLIC_LB: ApplicationLoadBalancer;
  constructor(scope: Construct, id: string, props: RbacStackProps) {
    super(scope, id, props);

    const prod = props.env!.account === PROD_ACCOUNT_ID;

    // Cluster
    const cluster = new Cluster(this, 'Cluster', {
      vpc: props.vpc,
    });
    Tag.add(cluster, 'ecs.Cluster', 'Rbac');

    // ASG
    const asg = cluster.addCapacity('Asg', {
      instanceType: new InstanceType('m5a.large'),
      minCapacity: prod ? 3 : 1,
      maxCapacity: prod ? 9 : 2,
      vpcSubnets: {
        subnetType: SubnetType.PRIVATE,
      },
    });
    asg.role.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));

    asg.scaleToTrackMetric('CpuReservation', {
      metric: cluster.metricCpuReservation(),
      targetValue: 80,
      cooldown: Duration.seconds(60),
      estimatedInstanceWarmup: Duration.seconds(60),
    });

    asg.scaleToTrackMetric('MemoryReservation', {
      metric: cluster.metricMemoryReservation(),
      targetValue: 80,
      cooldown: Duration.seconds(60),
      estimatedInstanceWarmup: Duration.seconds(60),
    });

    props.generalVpcESecurityGroup.connections.allowFrom(
      asg.connections,
      Port.tcp(443),
      'Allow connections from the RBAC cluster'
    );

    //#region setup up VPCe connetions
    props.vpcStack.dartVpcE.connections.allowFrom(
      asg.connections,
      Port.tcp(443),
      'Allow connections from the RBAC cluster'
    );
    props.vpcStack.cartmanVpcE.connections.allowFrom(
      asg.connections,
      Port.tcp(443),
      'Allow connections from the RBAC cluster'
    );
    props.vpcStack.discoveryVpcE.connections.allowFrom(
      asg.connections,
      Port.tcp(80),
      'Allow connections from the RBAC cluster'
    );
    props.vpcStack.owlVpcE.connections.allowFrom(
      asg.connections,
      Port.tcp(443),
      'Allow connections from the RBAC cluster'
    );
    props.vpcStack.userVpcE.connections.allowFrom(
      asg.connections,
      Port.tcp(443),
      'Allow connections from the RBAC cluster'
    );
    props.vpcStack.emsVpcE.connections.allowFrom(
      asg.connections,
      Port.tcp(443),
      'Allow connections from the RBAC cluster'
    );

    //#region Task
    const taskDefinition = new Ec2TaskDefinition(this, 'TaskDef');
    Tag.add(taskDefinition, 'ecs.TaskDefinition', 'Rbac');

    //#region Task Enviroment
    // TODO: Investigate which of these keys are actually important
    const taskEnv: { [key: string]: string } = {
      AWS_DEFAULT_REGION: props.env!.region!,
      APP: 'rbac',
      APP_DEPLOY_DIR: '/var/app/current', // I doubt this is used
      APP_STAGING_DIR: '/var/app/current', // ''
      CONFIG_PATH: 'config/production.json',
      ENVIRONMENT: 'production',
      GOPATH: '/var/app/current', // ''
      INFRA: 'ecs', // ''
      PATH: '/bin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin:/usr/local/go/bin:/var/app/current', // ''
      PORT: '9000', // Pretty sure this comes from the config
      RBAC_ENVIRONMENT: 'production', // ''
      RBAC_PGHOST: 'rbac-production-master.cbt09tmxkqne.us-west-2.rds.amazonaws.com', // ''
      RBAC_ROLLBARTOKEN: 'c898734c18fe4d0f8bcfdce0e6ceb463', // ''
      REGION: props.env!.region!,
      STATSD_HOST_PORT: 'graphite.internal.justin.tv:8125',
      TWITCH_ENV: 'production', // ''
    };

    if (!prod) {
      taskEnv.DEV = '1';
      taskEnv.ENVIRONMENT = 'staging';
      taskEnv.RBAC_ENVIRONMENT = 'staging';
      taskEnv.RBAC_PGHOST = 'rbac-staging-master.cfbymaguatde.us-west2.rds.amazonaws.com';
      taskEnv.RBAC_ROLLBARTOKEN = 'c898734c18fe4d0f8bcfdce0e6ceb463';
      taskEnv.TWITCH_ENV = 'staging';
      taskEnv.DEBUG = 'true';
    }

    //#endregion

    const taskContainer = taskDefinition.addContainer('rbac', {
      environment: taskEnv,
      image: ContainerImage.fromAsset(path.join(__dirname, '../../')),
      logging: new AwsLogDriver({
        streamPrefix: 'rbac',
        logRetention: 30,
      }),
      memoryLimitMiB: 1024, // TODO: Monitor this, prolly too high
    });

    taskContainer.addPortMappings({
      containerPort: 9000,
    });

    //#region Policy Statements
    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        effect: Effect.ALLOW,
        resources: ['*'],
      })
    );

    // New services must be onboarded to S2S/IAM Roles
    // https://dashboard.internal.justin.tv/s2s/services/
    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: ['sts:AssumeRole'],
        effect: Effect.ALLOW,
        resources: ['arn:aws:iam::180116294062:role/malachai/*'],
      })
    );

    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: ['sts:AssumeRole'],
        effect: Effect.ALLOW,
        resources: [prod ? PROD_PDMS_ROLE : DEV_PDMS_ROLE],
      })
    );

    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: ['execute-api:Invoke'],
        effect: Effect.ALLOW,
        resources: ['arn:aws:execute-api:*:985585625942:*'],
      })
    );

    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: [
          'logs:DescribeQueries',
          'logs:GetLogRecord',
          'logs:PutDestinationPolicy',
          'logs:StopQuery',
          'logs:TestMetricFilter',
          'logs:DeleteDestination',
          'logs:CreateLogGroup',
          'logs:GetLogDelivery',
          'logs:ListLogDeliveries',
          'logs:CreateLogDelivery',
          'logs:DeleteResourcePolicy',
          'logs:PutResourcePolicy',
          'logs:DescribeExportTasks',
          'logs:GetQueryResults',
          'logs:UpdateLogDelivery',
          'logs:CancelExportTask',
          'logs:DeleteLogDelivery',
          'logs:PutDestination',
          'logs:DescribeResourcePolicies',
          'logs:DescribeDestinations',
        ],
        effect: Effect.ALLOW,
        resources: ['*'],
      })
    );

    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: [
          'secretsmanager:GetResourcePolicy',
          'secretsmanager:GetSecretValue',
          'secretsmanager:DescribeSecret',
          'secretsmanager:ListSecretVersionIds',
        ],
        effect: Effect.ALLOW,
        resources: ['arn:aws:secretsmanager:*:*:secret:devsite/rbac/' + (prod ? 'production' : 'staging') + '/*'],
      })
    );

    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: ['sns:Publish'],
        effect: Effect.ALLOW,
        resources: [
          'arn:aws:sns:us-west-2:603200399373:pushy_darklaunch_dispatch',
          'arn:aws:sns:us-west-2:603200399373:pushy_production_dispatch',
        ],
      })
    );
    //#endregion Policy Statements
    //#endregion Task

    //#region New Service
    this.RBAC_SERVICE = new Ec2Service(this, 'Rbac', {
      cluster,
      taskDefinition,
      desiredCount: prod ? 3 : 1,
    });

    const rbacTaskAsg = this.RBAC_SERVICE.autoScaleTaskCount({
      maxCapacity: prod ? 12 : 3,
      minCapacity: prod ? 3 : 1,
    });

    rbacTaskAsg.scaleOnCpuUtilization('CpuScale', {
      targetUtilizationPercent: 80,
    });

    rbacTaskAsg.scaleOnMemoryUtilization('MemScale', {
      targetUtilizationPercent: 80,
    });

    const rbacSecurityGroup = new SecurityGroup(this, 'LB_SG', {
      vpc: props.vpc,
    });

    rbacSecurityGroup.addIngressRule(Peer.ipv4('10.0.0.0/8'), Port.tcp(443));
    rbacSecurityGroup.addIngressRule(Peer.ipv4('127.0.0.1/32'), Port.tcp(443)); // This shouldnt be needed but was part of the old rules.

    const rbacLB = new ApplicationLoadBalancer(this, 'PrivateLoadBalancer', {
      vpc: props.vpc,
      securityGroup: rbacSecurityGroup,
    });

    const rbacListener = rbacLB.addListener('Listener', {
      port: 443,
      certificates: [ListenerCertificate.fromCertificateManager(props.rbacCert)],
      open: false,
    });

    rbacListener.addTargets('Service', {
      port: 80,
      targets: [this.RBAC_SERVICE],
      healthCheck: {
        path: '/health',
      },
    });

    // Security Group -- Restricted to WPA2 using Amazon Prefix Lists (https://apll.corp.amazon.com/).
    const securityGroupAmazonVPN = new SecurityGroup(this, 'LoadBalancerPublicSecurityGroup', {
      vpc: props.vpc,
    });

    securityGroupAmazonVPN.addIngressRule(Peer.prefixList('pl-f8a64391'), Port.tcp(443));
    this.RBAC_PUBLIC_LB = new ApplicationLoadBalancer(this, 'PublicLoadBalancer', {
      vpc: props.vpc,
      securityGroup: securityGroupAmazonVPN,
      internetFacing: true,
    });
    const listenerPublic = this.RBAC_PUBLIC_LB.addListener('PublicListener', {
      port: 443,
      certificates: [ListenerCertificate.fromCertificateManager(props.rbacCert)],
      open: false,
    });

    listenerPublic.addTargets('Service', {
      port: 80,
      targets: [this.RBAC_SERVICE],
      healthCheck: {
        path: '/health',
      },
    });

    Tag.add(this.RBAC_SERVICE, 'ecs.Service', 'Rbac');

    //#endregion New Service
  }
}
