import { IVpc } from '@aws-cdk/aws-ec2';
import { PrivateHostedZone, ARecord, RecordTarget, HostedZone } from '@aws-cdk/aws-route53';
import { InterfaceVpcEndpointTarget, LoadBalancerTarget } from '@aws-cdk/aws-route53-targets';
import { PROD_ACCOUNT_ID, PROD_RBAC_A2Z, DEV_RBAC_A2Z, PROD_HOSTED_ZONE_ID, DEV_HOSTED_ZONE_ID } from './consts';
import { Stack, Construct, StackProps } from '@aws-cdk/core';
import { VpcStack } from './vpc-stack';
import { ApplicationLoadBalancer } from '@aws-cdk/aws-elasticloadbalancingv2';

interface RouteStackProps extends StackProps {
  vpc: IVpc;
  rbacPublicLB: ApplicationLoadBalancer;
}

export class RoutesStack extends Stack {
  public readonly DART_HOSTED_ZONE: PrivateHostedZone;
  public readonly CARTMAN_HOSTED_ZONE: PrivateHostedZone;
  public readonly DISCOVERY_HOSTED_ZONE: PrivateHostedZone;
  public readonly OWL_HOSTED_ZONE: PrivateHostedZone;
  public readonly USER_HOSTED_ZONE: PrivateHostedZone;
  public readonly EMS_HOSTED_ZONE: PrivateHostedZone;
  constructor(scope: Construct, name: string, props: RouteStackProps) {
    super(scope, name, props);

    const prod = props.env!.account === PROD_ACCOUNT_ID;
    this.DART_HOSTED_ZONE = new PrivateHostedZone(this, 'DartHostedZoneNew', {
      zoneName: prod ? 'us-west-2.prod.twitchdartreceiver.s.twitch.a2z.com' : 'us-west-2.beta.twitchdartreceiver.s.twitch.a2z.com',
      vpc: props.vpc,
    });

    this.CARTMAN_HOSTED_ZONE = new PrivateHostedZone(this, 'CartmanHostedZone', {
      zoneName: prod ? 'cartman.twitch.a2z.com' : 'cartman-dev.twitch.a2z.com',
      vpc: props.vpc,
    });

    this.DISCOVERY_HOSTED_ZONE = new PrivateHostedZone(this, 'DiscoveryHostedZone', {
      zoneName: prod ? 'prod.oldcategorydb.twitch.a2z.com' : 'staging.oldcategorydb.twitch.a2z.com',
      vpc: props.vpc,
    });

    this.OWL_HOSTED_ZONE = new PrivateHostedZone(this, 'OwlHostedZone', {
      zoneName: prod ? 'prod.owl.twitch.a2z.com' : 'staging.owl-dev.twitch.a2z.com',
      vpc: props.vpc,
    });

    this.USER_HOSTED_ZONE = new PrivateHostedZone(this, 'UserHostedZone', {
      zoneName: prod ? 'prod.users-service.twitch.a2z.com' : 'dev.users-service.twitch.a2z.com',
      vpc: props.vpc,
    });

    this.EMS_HOSTED_ZONE = new PrivateHostedZone(this, 'EmsHostedZone', {
      zoneName: prod ? 'prod.ems.extensions.twitch.a2z.com' : 'staging.ems.extensions.twitch.a2z.com',
      vpc: props.vpc,
    });

    const a2zZone = HostedZone.fromHostedZoneAttributes(this, 'RbacA2ZHostedZone', {
      zoneName: prod ? PROD_RBAC_A2Z : DEV_RBAC_A2Z,
      hostedZoneId: prod ? PROD_HOSTED_ZONE_ID : DEV_HOSTED_ZONE_ID,
    });

    new ARecord(this, 'RbacAmazonAlias', {
      zone: a2zZone,
      recordName: 'amzn',
      target: RecordTarget.fromAlias(new LoadBalancerTarget(props.rbacPublicLB)),
    });
  }

  // This should not be attached to the VPC stack, it should be in routes-stack, however CDK incorrectly exports the output for the IVPCE as an array instead of a string.
  addDartZone(vpcStack: VpcStack) {
    new ARecord(vpcStack, 'DartVpceAlias', {
      zone: this.DART_HOSTED_ZONE,
      target: RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(vpcStack.dartVpcE)),
    });

    new ARecord(vpcStack, 'CartmanVpceAlias', {
      zone: this.CARTMAN_HOSTED_ZONE,
      target: RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(vpcStack.cartmanVpcE)),
    });

    new ARecord(vpcStack, 'DiscoveryVpceAlias', {
      zone: this.DISCOVERY_HOSTED_ZONE,
      target: RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(vpcStack.discoveryVpcE)),
    });

    new ARecord(vpcStack, 'OwlVpceAlias', {
      zone: this.OWL_HOSTED_ZONE,
      target: RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(vpcStack.owlVpcE)),
    });

    new ARecord(vpcStack, 'UserVpceAlias', {
      zone: this.USER_HOSTED_ZONE,
      target: RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(vpcStack.userVpcE)),
    });

    new ARecord(vpcStack, 'EmsVpceAlias', {
      zone: this.EMS_HOSTED_ZONE,
      target: RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(vpcStack.emsVpcE)),
    });
  }
}
