package auth

import (
	"context"
	"strings"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

const (
	oauthPrefix     = "OAuth "
	oauthPrefixLen  = len("OAuth ")
	bearerPrefix    = "Bearer "
	bearerPrefixLen = len(bearerPrefix)
)

type ViennaRoleKey string

const ViennaUserWhitelistRoleKey ViennaRoleKey = "vienna-whitelist-role-key"

var (
	rawAuthorizationTokenKey = new(int)
	cartmanTokenKey          = new(int)
	userTwitchIDKey          = new(int)
)

// GetTwitchID returns the Twitch ID of the authenticated user, or "" if the request is anonymous.
// The value was set in the context by auth middleware, from the HTTP Authorization header.
func GetTwitchID(ctx context.Context) string {
	twitchID, ok := ctx.Value(userTwitchIDKey).(string)
	if !ok {
		return ""
	}
	return twitchID
}

// WithTwitchID adds the userTwitchID to the context, that can be retrieved later with GetTwitchID.
// Should only be used on unit tests.
func WithTwitchID(ctx context.Context, userTwitchID string) context.Context {
	return context.WithValue(ctx, userTwitchIDKey, userTwitchID)
}

// GetCartmanToken returns the value of the 'Twitch-Authorization' HTTP header, or an empty string if not found.
// This header is included in requests coming from Visage and GraphQL edges. Vienna (frontend) does not add this header.
func GetCartmanToken(ctx context.Context) string {
	cartmanToken, ok := ctx.Value(cartmanTokenKey).(string)
	if !ok {
		return ""
	}
	return cartmanToken
}

// WithCartmanToken adds the cartmanToken to the context, that can be retrieved later with GetCartmanToken.
// Should only be used on unit tests.
func WithCartmanToken(ctx context.Context, cartmanToken string) context.Context {
	return context.WithValue(ctx, cartmanTokenKey, cartmanToken)
}

// GetAuthorizationHeader returns the 'Authorization' HTTP header value that was used on this request,
// removing the "OAuth " or "Bearer " prefix. Returns an empty string if not found.
func GetAuthorizationToken(ctx context.Context) string {
	rawToken := GetRawAuthorizationToken(ctx)
	if strings.HasPrefix(rawToken, oauthPrefix) {
		return rawToken[oauthPrefixLen:]
	} else if strings.HasPrefix(rawToken, bearerPrefix) {
		return rawToken[bearerPrefixLen:]
	} else {
		return ""
	}
}

// GetRawAuthorizationToken returns the raw 'Authorization' HTTP header value that was used on this request.
func GetRawAuthorizationToken(ctx context.Context) string {
	v, ok := ctx.Value(rawAuthorizationTokenKey).(string)
	if !ok {
		return ""
	}
	return v
}

// WithRawAuthorizationToken adds the rawToken to the context, that can be retrieved later with
// GetRawAuthorizationToken. Should only be used on unit tests.
func WithRawAuthorizationToken(ctx context.Context, rawToken string) context.Context {
	return context.WithValue(ctx, rawAuthorizationTokenKey, rawToken)
}

func GetViennaWhitelistUserRole(ctx context.Context) *rbacrpc.WhitelistUserRole {
	userRole, ok := ctx.Value(ViennaUserWhitelistRoleKey).(rbacrpc.WhitelistUserRole)
	if !ok {
		return nil
	}

	return &userRole
}

func WithViennaWhitelistUserRole(ctx context.Context, role rbacrpc.WhitelistUserRole) context.Context {
	return context.WithValue(ctx, ViennaUserWhitelistRoleKey, role)
}

// IsRequestFromVienna checks if a request comes from Vienna by checking the oauth token in the context.
// Currently two places are calling RBAC APIs, Edge (Visage and GraphQL) and Vienna:
// Edge uses Cartman token (Twitch-Authorization header) while vienna uses OAuth (Authorization header).
// So if the request context contains an OAuth token, it means the request comes from Vienna.
func IsRequestFromVienna(ctx context.Context) bool {
	oauthToken := GetRawAuthorizationToken(ctx)
	return oauthToken != ""
}

func IsUserSecure(ctx context.Context) bool {
	if !IsRequestFromVienna(ctx) {
		return true // requests not from vienna come from the edge and company membership validation is used
	}
	role := GetViennaWhitelistUserRole(ctx)
	return role != nil && (*role == rbacrpc.WhitelistUserRole_SECURE_VIEWER || *role == rbacrpc.WhitelistUserRole_ADMIN)
}

// IsWhitelistAdmin returns true if the current user is a Vienna user or RBAC developer with admin access.
func IsWhitelistAdmin(ctx context.Context) bool {
	role := GetViennaWhitelistUserRole(ctx)
	return role != nil && *role == rbacrpc.WhitelistUserRole_ADMIN
}