package errorutil_test

import (
	"database/sql"
	"errors"
	"testing"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/foundation/twitchclient"
	"github.com/lib/pq"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestIsErrNoRows(t *testing.T) {
	require.True(t, errorutil.IsErrNoRows(sql.ErrNoRows))
	require.True(t, errorutil.IsErrNoRows(errx.Wrap(sql.ErrNoRows, "wrapped error")))

	require.False(t, errorutil.IsErrNoRows(errors.New("not what you are looking for")))
}

func TestIsErrUniqueViolation(t *testing.T) {
	require.True(t, errorutil.IsErrUniqueViolation(errorutil.NewErrUniqueViolation("already exists")))
	require.True(t, errorutil.IsErrUniqueViolation(&pq.Error{Code: "23505"}))
	require.True(t, errorutil.IsErrUniqueViolation(errx.Wrap(&pq.Error{Code: "23505"}, "wrapped error")))

	require.False(t, errorutil.IsErrUniqueViolation(sql.ErrNoRows))
	require.False(t, errorutil.IsErrUniqueViolation(&pq.Error{Code: "666"}))
	require.False(t, errorutil.IsErrUniqueViolation(errors.New("not what you are looking for")))
}

func TestIs(t *testing.T) {
	err1 := errors.New("plain error")
	err2 := twirp.NewError(twirp.NotFound, "twirp error")
	err3 := errx.Wrap(err1, "wrapped err1")
	err4 := errx.Wrap(err2, "wrapped err2")
	err5 := twirp.InternalErrorWith(err1)
	err6 := errx.Wrap(err5, "wrapped err5 and err1")

	require.True(t, errorutil.Is(err1, err1))
	require.True(t, errorutil.Is(err2, err2))
	require.True(t, errorutil.Is(err3, err3))
	require.True(t, errorutil.Is(err4, err4))
	require.True(t, errorutil.Is(err5, err5))
	require.True(t, errorutil.Is(err6, err6))

	require.True(t, errorutil.Is(err3, err1), "wrapped match")
	require.True(t, errorutil.Is(err4, err2), "wrapped match")
	require.True(t, errorutil.Is(err5, err1), "twirp.InternalErrorWith is also a wrapper")
	require.True(t, errorutil.Is(err6, err5), "wrapped match")
	require.True(t, errorutil.Is(err6, err1), "wrapped match, two levels down")

	require.False(t, errorutil.Is(nil, err1), "nil != err1")
	require.False(t, errorutil.Is(err1, nil), "err1 != nil")
	require.False(t, errorutil.Is(twirp.NewError(twirp.NotFound, "foo"), twirp.NewError(twirp.NotFound, "bar")))
	require.False(t, errorutil.Is(err5, err2), "err5 != err2")
	require.False(t, errorutil.Is(errors.New("same"), errors.New("same")), "same message, but different instances")
}

func TestUnwrap(t *testing.T) {
	err1 := errors.New("plain error")
	err2 := twirp.InternalErrorWith(err1)
	err3 := errx.Wrap(err2, "wrapped err1")

	require.Equal(t, err1, errorutil.Unwrap(err1), "unwrap plain error returns the same error")
	require.Equal(t, err1, errorutil.Unwrap(err2), "unwrap twirp.InternalErrorWith(err) returns the wrapped err")
	require.Equal(t, err2, errorutil.Unwrap(err3), "unwrap multiple wrapped returns only one level down")
	require.Nil(t, errorutil.Unwrap(nil), "unwrap nil returns nil")
}

func TestStatusCode(t *testing.T) {
	err1 := errors.New("plain error")
	err2 := twirp.InternalErrorWith(err1)
	err3 := errx.Wrap(err2, "wrapped err2")
	err4 := &twitchclient.Error{StatusCode: 401}
	err5 := errorutil.TwirpErrorFrom(err4)

	require.Equal(t, 0, errorutil.StatusCode(err1), "StatusCode of plain error is zero")
	require.Equal(t, 500, errorutil.StatusCode(err2), "Twirp Internal is a 500, even if wraps another error")
	require.Equal(t, 500, errorutil.StatusCode(err3), "Error wraps another error with status")
	require.Equal(t, 401, errorutil.StatusCode(err4), "twitchclient.Error StatusCode")
	require.Equal(t, 401, errorutil.StatusCode(err5), "Error wrapping twitchclient.Error")
}

func TestTwirpCodeFrom(t *testing.T) {
	err1 := errors.New("plain error")
	err2 := twirp.InternalErrorWith(err1)
	err3 := errx.Wrap(err2, "wrapped err2")
	err4 := &twitchclient.Error{StatusCode: 401}
	err5 := errorutil.TwirpErrorFrom(err4)
	err6 := twirp.NotFoundError("foobar")

	require.Equal(t, twirp.Internal, errorutil.TwirpCodeFrom(err1), "TwirpCodeFrom plain error is Internal")
	require.Equal(t, twirp.Internal, errorutil.TwirpCodeFrom(err2), "Twirp Internal, even if it wraps another error")
	require.Equal(t, twirp.Internal, errorutil.TwirpCodeFrom(err3), "Error wraps another error that is Internal")
	require.Equal(t, twirp.Unauthenticated, errorutil.TwirpCodeFrom(err4), "twitchclient.Error with status 401")
	require.Equal(t, twirp.Unauthenticated, errorutil.TwirpCodeFrom(err5), "Twirp wraps twitchclient.Error with 401")
	require.Equal(t, twirp.NotFound, errorutil.TwirpCodeFrom(err6), "Twirp error returns its own code")
}

func TestTwirpErrorFrom(t *testing.T) {
	err1 := errors.New("plain error")
	twerr1 := errorutil.TwirpErrorFrom(err1)
	require.Equal(t, twirp.Internal, twerr1.Code())
	require.Equal(t, "plain error", twerr1.Msg())
	require.Equal(t, err1, errorutil.Unwrap(twerr1), "wraps error")

	err2 := &twitchclient.Error{StatusCode: 401, Message: "seeya later yodaw"}
	twerr2 := errorutil.TwirpErrorFrom(err2)
	require.Equal(t, twirp.Unauthenticated, twerr2.Code())
	require.Equal(t, "seeya later yodaw", twerr2.Msg())
	require.Equal(t, err2, errorutil.Unwrap(twerr2), "wraps error")
}

func TestValidateUUID(t *testing.T) {
	invalidUUID := "not-a-uuid"
	err := errorutil.ValidateUUID("my_param", invalidUUID)
	require.EqualError(t, err, "twirp error invalid_argument: my_param invalid UUID format")
	require.Equal(t, twirp.InvalidArgument, errorutil.TwirpErrorFrom(err).Code())

	validUUID := common.NewUUID()
	noErr := errorutil.ValidateUUID("my_param", validUUID)
	require.NoError(t, noErr)
}

func TestValidateRequiredArgs(t *testing.T) {
	var err error
	zeroValArgs := errorutil.Args{
		{"zero-str", ""},
		{"zero-int", 0},
		{"zero-int32", int32(0)},
		{"zero-uint64", uint64(0)},
		{"zero-float64", float64(0.0)},
		{"zero-nil", nil},
		{"zero-error", err},
		{"zero-struct", struct{ Name string }{Name: ""}},
		{"zero-time", time.Time{}},
	}
	for _, arg := range zeroValArgs {
		err = errorutil.ValidateRequiredArgs(errorutil.Args{arg})
		require.EqualError(t, err, "twirp error invalid_argument: "+arg.Key+" is required")
	}

	err = errorutil.ValidateRequiredArgs(errorutil.Args{
		{"arg1", "value"},
		{"int arg 1", 66},
		{"zero-int-fails-here", 0}, // << fails here
		{"zero-string", ""},
	})
	require.EqualError(t, err, "twirp error invalid_argument: zero-int-fails-here is required")

	err = errorutil.ValidateRequiredArgs(nil)
	require.NoError(t, err, "valid with no values")

	err = errorutil.ValidateRequiredArgs(nil)
	require.NoError(t, err, "valid one value")

	err = errorutil.ValidateRequiredArgs(errorutil.Args{
		{"arg1", "value"},
		{"arg2", "non empty"},
		{"int arg 1", 66},
		{"", int32(99)}, // empty field key is fine
		{"structart", struct{ Name string }{Name: "foo"}},
	})
	require.NoError(t, err, "valid multiple values")
}
