package errorutil

import (
	"bytes"
	"encoding/json"
	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"github.com/sirupsen/logrus"
)

// prints each line of the errx.StackField on a new line
type StackTraceFormatter struct {
	Formatter logrus.Formatter
}

// rollbar.Stack is an internal package of errx
type stackFrame struct {
	Filename string `json:"filename"`
	Method   string `json:"method"`
	Line     int    `json:"lineno"`
}

func (f *StackTraceFormatter) Format(e *logrus.Entry) ([]byte, error) {
	rollbarStack, ok := e.Data[errx.StackField]
	if !ok {
		return f.Formatter.Format(e)
	}

	b, err := json.Marshal(rollbarStack)
	if err != nil {
		return f.Formatter.Format(e)
	}

	var stack []stackFrame
	if err := json.Unmarshal(b, &stack); err != nil {
		return f.Formatter.Format(e)
	}

	prettyPrintStack := func(stack []stackFrame) (string, error) {
		var prettyStack bytes.Buffer
		_, err := prettyStack.WriteString("\n")
		if err != nil {
			return "", err
		}
		for _, frame := range stack {
			path := f.filterFilename(frame.Filename)
			if path == "" {
				continue
			}
			if _, err := prettyStack.WriteString(fmt.Sprintf("%s:%d: %s\n", path, frame.Line, frame.Method)); err != nil {
				return "", err
			}
		}

		return prettyStack.String(), nil
	}

	prettyStack, err := prettyPrintStack(stack)
	if err != nil {
		return f.Formatter.Format(e)
	}

	e.Data[errx.StackField] = prettyStack
	return f.Formatter.Format(e)
}

func (f *StackTraceFormatter) filterFilename(filename string) string {
	if !strings.Contains(filename, "code.justin.tv/devrel/devsite-rbac") {
		return "" // skip stack frames from other libraries, they add too much noise
	}
	if strings.HasSuffix(filename, ".twirp.go") {
		return "" // skip autogenerated twirp methods
	}
	parts := strings.Split(filename, "code.justin.tv/devrel/devsite-rbac")
	return parts[1] // keep only the right part of the path
}
