package middleware

import (
	"context"

	"code.justin.tv/chat/golibs/logx"
	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
)

func RollbarHooks() *twirp.ServerHooks {
	hooks := &twirp.ServerHooks{}

	hooks.Error = func(ctx context.Context, twerr twirp.Error) context.Context {
		// 4xx errors are not logged (however we track response status in statsd in a different hook)
		// We must take the status code from twerr because that's what twirp returned.
		statusCode := errorutil.StatusCode(twerr)
		if statusCode < 500 && statusCode != 0 {
			return ctx
		}

		// Fields with info about the error and the context
		fields := logxErrorFields(ctx, twerr)

		// Non-twirp errors are wrapped with twirp.InternalErrorWith(err); try to unwrap
		// that error if possible, which in some cases is an errx.Error with StackTrace information.
		err := errorutil.Unwrap(twerr)

		// Log error in Rollbar
		logx.Error(ctx, err, fields)
		return ctx
	}

	return hooks
}

func LogxFields(ctx context.Context) logx.Fields {
	fields := make(logx.Fields)
	if service, ok := twirp.ServiceName(ctx); ok {
		fields["twirpService"] = service
	}
	if method, ok := twirp.MethodName(ctx); ok {
		fields["twirpMethod"] = method
	}
	if status, ok := twirp.StatusCode(ctx); ok {
		fields["twirpStatus"] = status
	}

	return fields
}

func logxErrorFields(ctx context.Context, twerr twirp.Error) logx.Fields {
	fields := LogxFields(ctx)
	fields["twirpErrorCode"] = twerr.Code()
	return fields
}
