package main

import (
	"code.justin.tv/commerce/splatter"
	"code.justin.tv/devrel/devsite-rbac/clients/pdms"
	"context"
	"fmt"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"goji.io/pat"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"
	"github.com/twitchtv/twirp"
	goji "goji.io"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/s2s2"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/gometrics"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/sse/malachai/pkg/events"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"

	"code.justin.tv/devrel/devsite-rbac/backend"
	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companyinvites"
	"code.justin.tv/devrel/devsite-rbac/backend/developerapplications"
	"code.justin.tv/devrel/devsite-rbac/backend/extensionreviewlogs"
	"code.justin.tv/devrel/devsite-rbac/backend/featuregating"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/viennauserwhitelist"
	"code.justin.tv/devrel/devsite-rbac/clients/cartman"
	"code.justin.tv/devrel/devsite-rbac/clients/channels_client"
	"code.justin.tv/devrel/devsite-rbac/clients/clue"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"
	"code.justin.tv/devrel/devsite-rbac/clients/discovery"
	"code.justin.tv/devrel/devsite-rbac/clients/evs"
	"code.justin.tv/devrel/devsite-rbac/clients/extensions"
	"code.justin.tv/devrel/devsite-rbac/clients/moneypenny"
	"code.justin.tv/devrel/devsite-rbac/clients/nioh"
	"code.justin.tv/devrel/devsite-rbac/clients/owlcli"
	"code.justin.tv/devrel/devsite-rbac/clients/passport"
	"code.justin.tv/devrel/devsite-rbac/clients/pushy"
	"code.justin.tv/devrel/devsite-rbac/clients/salesforce"
	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"code.justin.tv/devrel/devsite-rbac/config"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/internal/logstatter"
	"code.justin.tv/devrel/devsite-rbac/internal/middleware"
	"code.justin.tv/devrel/devsite-rbac/packagewrapper/localcache"
	"code.justin.tv/devrel/devsite-rbac/rpc/extensionreviewsserver"
	"code.justin.tv/devrel/devsite-rbac/rpc/privacyserver"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacactionhistoryserver"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacadminserver"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpcserver"
	"code.justin.tv/devrel/devsite-rbac/rpc/server"
)

func main() {
	conf := config.MustLoadFromFile()

	initLogxAndRollbar(conf)
	s2s2 := initS2S2(conf)
	stats := initStatsd(conf)
	discovery := initDiscovery(conf, stats)
	extensionsClient := initExtensionsClient(conf, stats)
	cartmanClient := initCartmanClient(conf, stats)
	emailer := initEmailer(conf, stats)
	evs := initEVS(conf, stats, s2s2)
	dart := initDart(conf, stats)
	salesforce := initSalesforce(conf, stats)
	owlClient := initOwlClient(conf, stats)
	users := initUsers(conf, stats)
	passport := initPassport(conf, stats)
	moneypenny := initMoneypenny(conf, stats)
	channels := initChannels(conf, stats)
	nioh := initNioh(conf, stats)
	clue := initClue(conf, stats)
	pdms := initPdms(conf, stats)

	dbxDB := common.MustConnectDBX(conf, stats)
	cache := initLocalCache(conf)
	dbBackend := initDBBackends(conf, dbxDB, stats, cache)

	hooks := twirp.ChainHooks(
		middleware.NewStatsdServerHooks(stats),
		middleware.RollbarHooks(),
		middleware.OwlHooks(owlClient),
		middleware.ViennaAuthWhitelistHooks(dbBackend.ViennaUserWhitelist),
	)

	rpc := rbacrpcserver.NewServer(&rbacrpcserver.Server{
		DropsJWTKey: conf.DropsJWTKey,

		// DB
		ActionHistories:     dbBackend.ActionHistories,
		Backend:             dbBackend.Backend,
		DevApps:             dbBackend.DeveloperApplications,
		FeatureGating:       dbBackend.FeatureGating,
		Memberships:         dbBackend.Memberships,
		CompanyInvites:      dbBackend.CompanyInvites,
		ViennaUserWhitelist: dbBackend.ViennaUserWhitelist,

		// Clients
		Cartman:    cartmanClient,
		Discovery:  discovery,
		Emailer:    emailer,
		EVS:        evs,
		Dart:       dart,
		Moneypenny: moneypenny,
		Owl:        owlClient,
		Passport:   passport,
		Salesforce: salesforce,
		Users:      users,
		Clue:       clue,
	}, hooks)

	rbacAdmin := rbacadminserver.New(&rbacadminserver.Server{
		Backend:             dbBackend.Backend,
		ActionHistories:     dbBackend.ActionHistories,
		ViennaUserWhitelist: dbBackend.ViennaUserWhitelist,
		FeatureGating:       dbBackend.FeatureGating,
		Memberships:         dbBackend.Memberships,
		CompanyInvites:      dbBackend.CompanyInvites,
		Users:               users,
		Channels:            channels,
		Nioh:                nioh,
		Clue:                clue,
		Passport:            passport,
	}, hooks)

	rbacPrivacy := privacyserver.New(&privacyserver.Server{
		ApiKey:				   conf.PdmsApiKey,
		Backend:               dbBackend.Backend,
		Memberships:           dbBackend.Memberships,
		DeveloperApplications: dbBackend.DeveloperApplications,
		ActionHistories:       dbBackend.ActionHistories,
		ViennaUserWhitelist:   dbBackend.ViennaUserWhitelist,
		PDMS:				   pdms,
	}, hooks)

	actionHistory := rbacactionhistoryserver.New(&rbacactionhistoryserver.Server{
		ActionHistories: dbBackend.ActionHistories,
	}, hooks)

	extensionReviews := extensionreviewsserver.New(&extensionreviewsserver.Server{
		Salesforce:            salesforce,
		Extensions:            extensionsClient,
		Cartman:               cartmanClient,
		Users:                 users,
		DBExtensionReviewLogs: dbBackend.ExtensionReviewLogs,
	}, hooks)

	mux := goji.NewMux()
	mux.Use(middleware.InterceptPanics)
	mux.Use(middleware.CORS(conf.Environment))
	mux.HandleFunc(pat.Get("/health"), runHealthCheck)
	mux.Use(middleware.AuthHeadersMiddleware)
	if calleeClient := initS2S(conf); calleeClient != nil {
		log.Println("using s2s middleware")
		mux.Use(middleware.S2S(calleeClient))
	}
	for _, route := range []*server.Route{
		rpc, rbacAdmin, actionHistory, extensionReviews, rbacPrivacy,
	} {
		mux.Handle(route.Pattern, route.Handler)
	}

	log.Printf("serving rpc at %s", conf.RPCHostPort)
	// Gracefully shutdown the service.
	// This is _needed_ for e2e coverage to work. The program needs to exit
	// gracefully for coverage to be reported.
	srv := &http.Server{Addr: conf.RPCHostPort, Handler: mux}
	idleConnsClosed := make(chan struct{})
	go func() {
		sigint := make(chan os.Signal, 1)
		signal.Notify(sigint, syscall.SIGINT, syscall.SIGTERM)
		log.Println("Waiting for shutdown")
		<-sigint
		log.Println("Shutdown received")

		// We received an interrupt signal, shut down.
		if err := srv.Shutdown(context.Background()); err != nil {
			// Error from closing listeners, or context timeout:
			log.Printf("HTTP server Shutdown: %v", err)
		}
		close(idleConnsClosed)

		log.Println("Shutdown complete")
	}()

	if err := srv.ListenAndServe(); err != http.ErrServerClosed {
		// Error starting or closing listener:
		log.Printf("HTTP server ListenAndServe: %v", err)
	}

	<-idleConnsClosed
}

func initLogxAndRollbar(conf *config.RBAC) {
	// Set log level to ERROR (ignore WARN, INFO and DEBUG)
	logrus.SetLevel(logrus.ErrorLevel) // logrus default logger is used by logx
	if os.Getenv("DEBUG") != "" {
		log.Println("setting log level to info")
		logrus.SetLevel(logrus.InfoLevel) // Info on debug mode (as low as logx goes)
	}

	// Configure logx.Error with Rollbar
	logx.InitDefaultLogger(logx.Config{
		RollbarToken: conf.RollbarToken,
		RollbarEnv:   conf.Environment,
	})

	if conf.Environment == "e2e" {
		// Let logrus print the errx stacktrace field that's filtered by logx.InitDefaultLogger
		logrus.SetFormatter(&errorutil.StackTraceFormatter{
			Formatter: logx.DefaultFormatter(),
		})
	}
}
func initS2S2(conf *config.RBAC) *s2s2.S2S2 {
	if conf.S2SService == "" {
		return nil
	}

	s2s2Client, err := s2s2.New(&s2s2.Options{
		Config: &c7s.Config{
			ClientServiceName: conf.S2SService,
		},
	})
	if err != nil {
		log.Fatalf("creating s2s2 client: %q", err)
	}

	return s2s2Client
}

func initStatsd(conf *config.RBAC) statsd.Statter {
	if conf.StatsdHostPort == "NOOP" {
		return &statsd.NoopClient{} // noop
	}
	if conf.StatsdHostPort == "LOGSTDOUT" {
		return &logstatter.LogStatter{} // show stats in console
	}

	host, err := os.Hostname()
	if err != nil {
		host = "unknown"
	}

	prefix := fmt.Sprintf("twitch-rbac.%s.%s", conf.Environment, host)

	buffConfig := &splatter.BufferedTelemetryConfig{
		FlushPeriod:       30 * time.Second,
		BufferSize:        10000,
		AggregationPeriod: time.Minute,
		ServiceName:       "RBAC",
		AWSRegion:         "us-west-2",
		Stage:             "production",
		Prefix:            prefix,
	}
	telStat := splatter.NewBufferedTelemetryCloudWatchStatter(buffConfig, map[string]bool{})

	gometrics.Monitor(telStat, 5*time.Second)
	// Monitor Go application metrics

	return telStat
}

func initSalesforce(c *config.RBAC, stats statsd.Statter) salesforce.Client {
	if c.SalesforceURL == "NOOP" {
		return &salesforce.NoopClient{}
	}

	client, err := salesforce.NewSalesforceClient(salesforce.Config{
		Host:          c.SalesforceURL,
		Username:      c.SalesforceUsername,
		ClientID:      c.SalesforceClientID,
		RSA256PrivKey: c.SalesforceRSA256PrivKey,
		Stats:         stats,
	})
	if err != nil {
		log.Fatalf("Unable to create salesforce client: %q", err)
	}

	return client
}

type DBBackends struct {
	Backend backend.Backender // one size fits all big interface, hopefully we start splitting away into smaller models

	ActionHistories       actionhistories.ActionHistories
	Memberships           memberships.Memberships
	CompanyInvites        companyinvites.CompanyInvites
	DeveloperApplications developerapplications.DeveloperApplications
	ExtensionReviewLogs   extensionreviewlogs.ExtensionReviewLogs
	ViennaUserWhitelist   viennauserwhitelist.UserWhitelist
	FeatureGating         featuregating.FeatureGating
}

func initDBBackends(conf *config.RBAC, dbxDB common.DBXer, stats statsd.Statter, cache localcache.LocalCache) DBBackends {
	backend, err := backend.NewBackend(&backend.Config{DB: dbxDB, Stats: stats})
	if err != nil {
		log.Fatalf("unable to initialize backend: %s", err.Error())
	}

	actionHistories := actionhistories.New(dbxDB, stats)
	dbMemberships := memberships.New(dbxDB, stats)
	dbCompanyInvites := companyinvites.New(dbxDB, stats)
	extensionReviewLogs := extensionreviewlogs.New(dbxDB, stats)
	viennaUserWhitelist := viennauserwhitelist.New(dbxDB, stats)
	developerApplications := developerapplications.New(dbxDB, stats)
	featureGating := featuregating.New(dbxDB, stats, cache)

	return DBBackends{
		Backend:               backend,
		ActionHistories:       actionHistories,
		Memberships:           dbMemberships,
		CompanyInvites:        dbCompanyInvites,
		DeveloperApplications: developerApplications,
		ExtensionReviewLogs:   extensionReviewLogs,
		ViennaUserWhitelist:   viennaUserWhitelist,
		FeatureGating:         featureGating,
	}
}

func initDiscovery(conf *config.RBAC, stats statsd.Statter) discovery.Discovery {
	if conf.DiscoveryHost == "NOOP" {
		// end-to-end tests use a NOOP discovery client to ensure reliable results
		return &discovery.NoopClient{}
	}

	client, err := discovery.NewDiscovery(twitchclient.ClientConf{
		Host:  conf.DiscoveryHost,
		Stats: stats,
	})
	if err != nil {
		log.Fatalf("unable to initialize discovery: %q", err)
	}
	return client
}

func initEmailer(conf *config.RBAC, stats statsd.Statter) pushy.Emailer {
	if conf.PushySNSTopic == "NOOP" {
		return &pushy.NoopClient{}
	}

	client, err := pushy.New(conf.PushySNSTopic, stats)
	if err != nil {
		log.Fatalf("unable to initialize pushy client: %q", err)
	}
	return client
}

func initExtensionsClient(conf *config.RBAC, stats statsd.Statter) extensions.Client {
	if conf.ExtensionsEMSHost == "NOOP" {
		return &extensions.FakeClient{}
	}
	return extensions.NewClient(twitchclient.ClientConf{
		Host:  conf.ExtensionsEMSHost,
		Stats: stats,
	})
}

func initCartmanClient(conf *config.RBAC, stats statsd.Statter) cartman.Client {
	if conf.CartmanHost == "NOOP" {
		return &cartman.NoopClient{}
	}
	return cartman.NewClient(twitchclient.ClientConf{
		Host:  conf.CartmanHost,
		Stats: stats,
	})
}

func initEVS(conf *config.RBAC, stats statsd.Statter, s2s2 *s2s2.S2S2) evs.EVS {
	if conf.EVSHost == "NOOP" {
		return &evs.NoopClient{}
	}

	// setup the evs client
	client, err := evs.NewEVS(
		conf.EVSHost,
		twitchclient.ClientConf{
			Host:  conf.EVSHost,
			Stats: stats,
		},
		s2s2,
	)
	if err != nil {
		log.Fatalf("unable to initialize evs client: %q", err)
	}
	return client
}

func initDart(conf *config.RBAC, stats statsd.Statter) dart.Dart {
	if conf.DartHost == "NOOP" {
		return &dart.NoopClient{}
	}

	client, err := dart.New(conf.DartHost)
	if err != nil {
		log.Fatalf("unable to initialize dart client: %q", err)
	}
	return client
}

func initOwlClient(conf *config.RBAC, stats statsd.Statter) owlcli.Client {
	if conf.OwlHost == "NOOP" {
		// end-to-end tests use a fake discovery client to ensure reliable results
		return &owlcli.Fake{}
	}

	client, err := owlcli.NewClient(twitchclient.ClientConf{
		Host:  conf.OwlHost,
		Stats: stats,
	})
	if err != nil {
		log.Fatalf("unable to initialize owl client: %q", err)
	}
	return client
}

func initUsers(conf *config.RBAC, stats statsd.Statter) users.Users {
	if conf.UsersHost == "NOOP" {
		return &users.NoopClient{}
	}

	client, err := users.NewUsers(twitchclient.ClientConf{
		Host:  conf.UsersHost,
		Stats: stats,
	})
	if err != nil {
		log.Fatalf("unable to initialize users client: %q", err)
	}
	return client
}

func initChannels(conf *config.RBAC, stats statsd.Statter) channels_client.Channels {
	if conf.UsersHost == "NOOP" {
		return &channels_client.NoopClient{}
	}

	client, err := channels_client.NewChannels(twitchclient.ClientConf{
		Host:  conf.UsersHost,
		Stats: stats,
	})
	if err != nil {
		log.Fatalf("unable to initialize channels client: %q", err)
	}
	return client
}

func initLocalCache(conf *config.RBAC) localcache.LocalCache {
	defaultExpirationTime := time.Duration(conf.LocalCacheExpirationTimeInSec * int64(time.Second))
	defaultCleanUpInterval := time.Duration(conf.LocalCacheCleanUpIntervalInSec * int64(time.Second))
	return localcache.InitLocalCache(defaultExpirationTime, defaultCleanUpInterval)
}

func initPassport(conf *config.RBAC, stats statsd.Statter) passport.Client {
	if conf.PassportHost == "NOOP" {
		return passport.Fake{}
	}

	client, err := passport.NewClient(twitchclient.ClientConf{
		Host:  conf.PassportHost,
		Stats: stats,
	})
	if err != nil {
		log.Fatalf("unable to initialize passport client: %q", err)
	}
	return client
}

func initMoneypenny(conf *config.RBAC, stats statsd.Statter) moneypenny.Client {
	if conf.MoneypennyHost == "NOOP" {
		return moneypenny.Fake{}
	}

	client, err := moneypenny.NewClient(twitchclient.ClientConf{
		Host:  conf.MoneypennyHost,
		Stats: stats,
	})
	if err != nil {
		log.Fatalf("unable to initialize moneypenny client: %q", err)
	}
	return client
}

func initClue(conf *config.RBAC, stats statsd.Statter) clue.Clue {
	if conf.ClueHost == "NOOP" {
		return &clue.NoopClient{}
	}

	return clue.NewClient(conf.ClueHost, twitchclient.ClientConf{
		Host:  conf.ClueHost,
		Stats: stats,
	})
}

func initNioh(conf *config.RBAC, stats statsd.Statter) nioh.Nioh {
	if conf.NiohHost == "NOOP" {
		return &nioh.NoopClient{}
	}

	return nioh.NewClient(conf.NiohHost, twitchclient.ClientConf{
		Host:  conf.NiohHost,
		Stats: stats,
	})
}

func initPdms(conf *config.RBAC, stats statsd.Statter) pdms.PDMS {
	if conf.PdmsCallerRole == "NOOP" {
		return &pdms.NoopClient{}
	}

	return pdms.NewClient(conf.PdmsCallerRole, conf.PdmsLambdaArn)
}

func runHealthCheck(w http.ResponseWriter, r *http.Request) {
	_, err := w.Write([]byte("OK"))
	if err != nil {
		logx.Error(r.Context(), errx.Wrap(err, "failed to write /health response"))
	}
}

// initS2S best effort initializes the s2s callee client because
// s2s is running in pass through mode. when our callers are onboarded
// to s2s and whitelisted to us, we should fail RBAC if initialize fails
func initS2S(conf *config.RBAC) *callee.Client {
	ctx := context.Background()
	if conf.S2SService == "" {
		return nil
	}

	eventWriter, err := events.NewEventLogger(events.Config{}, nil)
	if err != nil {
		logx.Error(ctx, errors.Wrap(err, "failed initialize s2s event logger"))
		return nil
	}

	calleeClient := &callee.Client{
		ServiceName:        conf.S2SService,
		EventsWriterClient: eventWriter,
		Config: &callee.Config{
			PassthroughMode: true,
		},
	}
	if err := calleeClient.Start(); err != nil {
		logx.Error(ctx, errors.Wrap(err, "failed to start s2s callee client"))
		return nil
	}

	return calleeClient
}
