package extensionreviewsserver

import (
	"errors"
	"testing"

	"github.com/stretchr/testify/require"

	"code.justin.tv/devrel/devsite-rbac/backend/extensionreviewlogs/extensionreviewlogsfakes"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/foundation/twitchclient"
)

func TestSetStateApproved_Success(t *testing.T) {
	server, _ := NewTestServer()
	dbReviewLogs := server.DBExtensionReviewLogs.(*extensionreviewlogsfakes.FakeExtensionReviewLogs)
	dbReviewLogs.GetLastInReviewReturns(&rbacrpc.ExtensionReviewLog{}, nil)

	_, err := server.SetStateApproved(ctxWithTwitchID, newSetStateReviewerRequest())
	require.NoError(t, err)
}

func TestSetStateApproved_ValidatesParams(t *testing.T) {
	var err error
	server, _ := NewTestServer()

	_, err = server.SetStateApproved(ctxWithTwitchID, &rbacrpc.SetStateReviewerRequest{})
	require.EqualError(t, err, "twirp error invalid_argument: extension_id is required")

	_, err = server.SetStateApproved(ctxWithTwitchID, &rbacrpc.SetStateReviewerRequest{ExtensionId: "fff"})
	require.EqualError(t, err, "twirp error invalid_argument: extension_version is required")
}

func TestSetStateApproved_RequiresAuthorizationHeader(t *testing.T) {
	server, _ := NewTestServer()

	_, err := server.SetStateApproved(ctx, newSetStateReviewerRequest())
	require.EqualError(t, err, "twirp error unauthenticated: Mising Authorization header with OAuth token")
}

func TestSetStateApproved_EMSStateTransitionError(t *testing.T) {
	server, extensions := NewTestServer()
	extensions.TransitionExtensionStateReturnsError = &twitchclient.Error{StatusCode: 500, Message: "oops"}

	_, err := server.SetStateApproved(ctxWithTwitchID, newSetStateReviewerRequest())
	require.EqualError(t, err, "twirp error internal: oops")
}

func TestSetStateApproved_IgnoresReviewLogsError(t *testing.T) {
	server, _ := NewTestServer()
	dbReviewLogs := server.DBExtensionReviewLogs.(*extensionreviewlogsfakes.FakeExtensionReviewLogs)
	dbReviewLogs.InsertReturns(errors.New("Ignore me"))

	_, err := server.SetStateApproved(ctxWithTwitchID, newSetStateReviewerRequest())
	require.NoError(t, err) // ignored
}

//
// Test helpers
//

func newSetStateReviewerRequest() *rbacrpc.SetStateReviewerRequest {
	return &rbacrpc.SetStateReviewerRequest{
		ExtensionId:       "fff",
		ExtensionVersion:  "0.0.1'",
		ReviewReasonCodes: []string{"1.1", "6.2.2"},
		ReviewReason:      "Stuff happens",
	}
}
