package extensionreviewsserver

import (
	"context"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/clients/salesforce"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/foundation/twitchclient"
	"github.com/twitchtv/twirp"
)

const (
	authorChannelStringSizeLimit = 64
)

func (s *Server) SetStateInReview(ctx context.Context, params *rbacrpc.SetStateInReviewRequest) (*rbacrpc.Empty, error) {
	if err := validateSetStateInReviewRequest(params); err != nil {
		return nil, err
	}

	// Cartman Token header. This endpoint should only be called from the edge (GraphQL),
	// with already authenticated user that is encoded in the Twitch-Authorization header, required by EMS (Extensions).
	cartmanToken := auth.GetCartmanToken(ctx)
	if cartmanToken == "" {
		return nil, twirp.NewError(twirp.Unauthenticated, "Cartman Twitch-Authorization header is required")
	}

	// Transition to "review" in EMS backend
	err := s.Extensions.TransitionExtensionState(ctx, params.ExtensionId, params.ExtensionVersion, "review", cartmanToken)
	if err != nil {
		twerr := errorutil.TwirpErrorFrom(err)
		if tcerr, ok := errorutil.Unwrap(err).(*twitchclient.Error); ok {
			twerr = twerr.WithMeta("error_code", tcerr.ErrorCode) //  some EMS errors have an error_code, include it as meta string
		}
		return nil, twerr
	}

	// Load extension first to make sure it exists, and to get extra fields for the Salesforce case.
	extv, err := s.Extensions.GetExtensionVersion(ctx, params.ExtensionId, params.ExtensionVersion, cartmanToken)
	if err != nil {
		return nil, errorutil.TwirpErrorFrom(err)
	}
	extName := extv.Name

	// Load user data for the Salesforce case.
	user, err := s.Users.GetUserByID(ctx, params.UserTwitchId)
	if err != nil {
		return nil, errorutil.TwirpErrorFrom(err)
	}
	userLogin := "nil"
	if user.Login != nil {
		userLogin = *user.Login
	}
	userEmail := "nil@nil.nil"
	if user.Email != nil {
		userEmail = *user.Email
	}

	// Create a review Case on Salesforce.
	salesforceID, err := s.Salesforce.CreateExtensionReviewCase(ctx, salesforce.ExtensionReviewCase{
		Origin:            "Vienna",
		Type:              "Extension",
		RecordTypeID:      "0125A000001AoODQA0", // DX Review type, since SalesForce is shared, denotes which layouts and such
		Subject:           "Review: " + extName + " - " + params.ExtensionVersion,
		Description:       params.Notes,
		ContactEmail:      userEmail,
		SuppliedEmail:     "todo-pending@extensions.team",
		TwitchUserID:      params.UserTwitchId,
		TwitchLogin:       userLogin,
		ExtensionID:       params.ExtensionId,
		ExtensionVersion:  params.ExtensionVersion,
		TwitchTestChannel: params.TestChannel,
	})
	if err != nil {
		// If Salesforce fails, we don't want to prevent the user from setting the extension in review.
		// The extension-version state is updated in EMS and Termina (moderation tool) uses that a source of thruth,
		// they will know that the Salesforce case was not created and take manual steps to fix that.
		logx.Error(ctx, err, logx.Fields{
			"user_id":           params.UserTwitchId,
			"extension_id":      params.ExtensionId,
			"extension_version": params.ExtensionVersion,
			"extension_name":    extName,
		})
		salesforceID = ""
	}

	// Log transition with channel and notes
	err = s.DBExtensionReviewLogs.Insert(ctx, &rbacrpc.ExtensionReviewLog{
		CreatedAt:        time.Now().Format(time.RFC3339),
		State:            "review",
		AuthorTwitchId:   params.UserTwitchId,
		AuthorNotes:      params.Notes,
		AuthorChannel:    params.TestChannel,
		ExtensionId:      params.ExtensionId,
		ExtensionVersion: params.ExtensionVersion,
		SalesforceCaseId: salesforceID,
	})
	if err != nil {
		// Don't fail the request, because the extension is already in reivew and the logs are not critical.
		// However, make sure we report the error and get paged by this because something may be wrong with the DB.
		logx.Error(ctx, err, logx.Fields{
			"AuthorTwitchId":   params.UserTwitchId,
			"AuthorChannel":    params.TestChannel,
			"ExtensionId":      params.ExtensionId,
			"ExtensionVersion": params.ExtensionVersion,
			"SalesforceCaseId": salesforceID,
		})
	}

	return &rbacrpc.Empty{}, nil
}

func validateSetStateInReviewRequest(params *rbacrpc.SetStateInReviewRequest) error {
	if params.UserTwitchId == "" {
		return twirp.RequiredArgumentError("user_twitch_id")
	}
	if params.ExtensionId == "" {
		return twirp.RequiredArgumentError("extension_id")
	}
	if params.ExtensionVersion == "" {
		return twirp.RequiredArgumentError("extension_version")
	}
	if params.TestChannel == "" {
		return twirp.RequiredArgumentError("test_channel")
	}

	if len(params.TestChannel) > authorChannelStringSizeLimit {
		return twirp.InvalidArgumentError("test_channel", "string size exceeds 64 characters.")
	}

	if params.Notes == "" {
		return twirp.RequiredArgumentError("notes")
	}
	return nil
}
