package extensionreviewsserver

import (
	"errors"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/extensionreviewlogs/extensionreviewlogsfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/salesforce"
	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestSetStateInReview_Success(t *testing.T) {
	server, _ := NewTestServer()

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	require.NoError(t, err)
}

func TestSetStateInReview_ValidatesParams(t *testing.T) {
	var err error
	server, _ := NewTestServer()

	_, err = server.SetStateInReview(ctxWithCartmanToken, &rbacrpc.SetStateInReviewRequest{})
	require.EqualError(t, err, "twirp error invalid_argument: user_twitch_id is required")

	_, err = server.SetStateInReview(ctxWithCartmanToken, &rbacrpc.SetStateInReviewRequest{
		UserTwitchId: "123"})
	require.EqualError(t, err, "twirp error invalid_argument: extension_id is required")

	_, err = server.SetStateInReview(ctxWithCartmanToken, &rbacrpc.SetStateInReviewRequest{
		UserTwitchId: "123", ExtensionId: "fff"})
	require.EqualError(t, err, "twirp error invalid_argument: extension_version is required")

	_, err = server.SetStateInReview(ctxWithCartmanToken, &rbacrpc.SetStateInReviewRequest{
		UserTwitchId: "123", ExtensionId: "fff", ExtensionVersion: "0.0.1"})
	require.EqualError(t, err, "twirp error invalid_argument: test_channel is required")

	_, err = server.SetStateInReview(ctxWithCartmanToken, &rbacrpc.SetStateInReviewRequest{
		UserTwitchId: "123", ExtensionId: "fff", ExtensionVersion: "0.0.1", TestChannel: "foobar"})
	require.EqualError(t, err, "twirp error invalid_argument: notes is required")
}

func TestSetStateInReview_RequiresCartmanTokenHeader(t *testing.T) {
	server, _ := NewTestServer()

	_, err := server.SetStateInReview(ctx, newSetStateInReviewRequest())
	require.EqualError(t, err, "twirp error unauthenticated: Cartman Twitch-Authorization header is required")
}

func TestSetStateInReview_GetExtensionError(t *testing.T) {
	server, extensions := NewTestServer()
	extensions.GetExtensionReturnsError = &twitchclient.Error{StatusCode: 404, Message: "Can not get extension"}

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	require.EqualError(t, err, "twirp error not_found: Can not get extension")
}

func TestSetStateInReview_EMSFailureWithErrorCode(t *testing.T) {
	server, extensions := NewTestServer()
	extensions.TransitionExtensionStateReturnsError = &twitchclient.Error{
		StatusCode: 400,
		Message:    "Can not transition state",
		ErrorCode:  "multiple_extensions_in_review",
	}

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	twerr, ok := err.(twirp.Error)
	require.True(t, ok, "Expected a Twirp error")
	require.Equal(t, "Can not transition state", twerr.Msg())
	require.Equal(t, twirp.InvalidArgument, twerr.Code())
	require.Equal(t, "multiple_extensions_in_review", twerr.Meta("error_code")) // included in metadata
}

func TestSetStateInReview_EMSUnprocessableEntityErrors(t *testing.T) {
	server, extensions := NewTestServer()
	extensions.TransitionExtensionStateReturnsError = &twitchclient.Error{
		StatusCode: 422,
		Message:    "You can't transition to in_review without specifying the support email first",
		ErrorCode:  "transition_missing_category",
	}

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	twerr, ok := err.(twirp.Error)
	require.True(t, ok, "Expected a Twirp error")
	require.Equal(t, "You can't transition to in_review without specifying the support email first", twerr.Msg())
	require.Equal(t, "transition_missing_category", twerr.Meta("error_code"))
	require.Equal(t, twirp.InvalidArgument, twerr.Code()) // 422 is conveted to InvalidArgument
}

func TestSetStateInReview_UsersServiceError(t *testing.T) {
	server, _ := NewTestServer()
	users := server.Users.(*users.NoopClient)
	users.GetUserByIDReturnsError = errors.New("network down")

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	twerr, ok := err.(twirp.Error)
	require.True(t, ok, "Expected a Twirp error")
	require.Equal(t, "network down", twerr.Msg())
	require.Equal(t, twirp.Internal, twerr.Code())
}

func TestSetStateInReview_IgnoresReviewLogsError(t *testing.T) {
	server, _ := NewTestServer()
	dbReviewLogs := server.DBExtensionReviewLogs.(*extensionreviewlogsfakes.FakeExtensionReviewLogs)
	dbReviewLogs.InsertReturns(errors.New("Ignore me"))

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	require.NoError(t, err) // ignored
}

func TestSetStateInReview_IgnoresSalesforceError(t *testing.T) {
	server, _ := NewTestServer()
	salesforce := server.Salesforce.(*salesforce.NoopClient)
	salesforce.CreateExtensionReviewCaseReturnsError = errors.New("Ignore me")

	_, err := server.SetStateInReview(ctxWithCartmanToken, newSetStateInReviewRequest())
	require.NoError(t, err) // ignored
}

//
// Test helpers
//

func newSetStateInReviewRequest() *rbacrpc.SetStateInReviewRequest {
	return &rbacrpc.SetStateInReviewRequest{
		UserTwitchId:     "123",
		ExtensionId:      "fff",
		ExtensionVersion: "0.0.1",
		TestChannel:      "http://twitch.tv/tothemario",
		Notes:            "Foobar notes",
	}
}
