package privacyserver

import (
	"context"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"github.com/stretchr/testify/require"
)

func TestProcessDeletionRequest_HasNoCompany_Success(t *testing.T) {
	testPrivacyServer := NewTestPrivacyServer()
	twitchID := "123"

	testPrivacyServer.MockMemberships.GetMembershipsByAccountReturns([]memberships.Membership{}, nil)
	_, err := testPrivacyServer.Server.ProcessDeletionRequest(context.Background(), twitchID)
	require.NoError(t, err)

	// these should be called
	require.Equal(t, 1, testPrivacyServer.MockMemberships.GetMembershipsByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockViennaUserWhitelist.RemoveUserFromWhitelistCallCount())
	require.Equal(t, 1, testPrivacyServer.MockActionHistories.StripUserActionHistoryCallCount())
	require.Equal(t, 1, testPrivacyServer.MockDeveloperApplications.DeleteDeveloperApplicationByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteCompanyApplicationsByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteAllCompanyInvitesByTwitchIdCallCount())

	// this should not be called due to user having no membership
	require.Equal(t, 0, testPrivacyServer.MockMemberships.DeleteMembershipCallCount())
}

func TestProcessDeletionRequest_HasCompanyIsDeveloper_Success(t *testing.T) {
	testPrivacyServer := NewTestPrivacyServer()
	twitchID := "123"

	uuid := common.NewUUID()

	membership := memberships.Membership{
		CompanyID: uuid,
		Role:      "Developer",
		FirstName: "McDudeCool",
	}

	testPrivacyServer.MockMemberships.GetMembershipsByAccountReturns([]memberships.Membership{membership}, nil)
	_, err := testPrivacyServer.Server.ProcessDeletionRequest(context.Background(), twitchID)

	require.NoError(t, err)

	// these should be called
	require.Equal(t, 1, testPrivacyServer.MockMemberships.GetMembershipsByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockMemberships.DeleteMembershipCallCount())
	require.Equal(t, 1, testPrivacyServer.MockViennaUserWhitelist.RemoveUserFromWhitelistCallCount())
	require.Equal(t, 1, testPrivacyServer.MockActionHistories.StripUserActionHistoryCallCount())
	require.Equal(t, 1, testPrivacyServer.MockDeveloperApplications.DeleteDeveloperApplicationByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteCompanyApplicationsByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteAllCompanyInvitesByTwitchIdCallCount())

	// these should be called since the user is developer (not an owner)
	require.Equal(t, 0, testPrivacyServer.MockMemberships.GetAdminsByCompanyCallCount())
	require.Equal(t, 0, testPrivacyServer.MockMemberships.MigrateOwnerCallCount())
}
func TestProcessDeletionRequest_HasCompanyIsOwner_Success(t *testing.T) {
	testPrivacyServer := NewTestPrivacyServer()
	twitchID := "123"

	membership := memberships.Membership{
		CompanyID: common.NewUUID(),
		Role:      "Owner",
		FirstName: "McDudeCool",
	}

	membership2 := memberships.Membership{
		CompanyID: common.NewUUID(),
		Role:      "Administrator",
		FirstName: "McDudeCool",
	}

	ctx := context.Background()
	testPrivacyServer.MockMemberships.GetMembershipsByAccountReturns([]memberships.Membership{membership}, nil)
	testPrivacyServer.MockMemberships.GetAdminsByCompanyReturns([]memberships.Membership{membership2}, nil)
	testPrivacyServer.MockBackender.BeginReturns(ctx)
	_, err := testPrivacyServer.Server.ProcessDeletionRequest(ctx, twitchID)

	require.NoError(t, err)

	// these should be called
	require.Equal(t, 1, testPrivacyServer.MockMemberships.GetMembershipsByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockMemberships.DeleteMembershipCallCount())
	require.Equal(t, 1, testPrivacyServer.MockViennaUserWhitelist.RemoveUserFromWhitelistCallCount())
	require.Equal(t, 1, testPrivacyServer.MockActionHistories.StripUserActionHistoryCallCount())
	require.Equal(t, 1, testPrivacyServer.MockDeveloperApplications.DeleteDeveloperApplicationByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteCompanyApplicationsByAccountCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteAllCompanyInvitesByTwitchIdCallCount())

	// these should ALSO be called since the user is owner
	require.Equal(t, 1, testPrivacyServer.MockMemberships.GetAdminsByCompanyCallCount())
	require.Equal(t, 1, testPrivacyServer.MockActionHistories.InsertActionHistoryCallCount())
	require.Equal(t, 1, testPrivacyServer.MockMemberships.MigrateOwnerCallCount())
}

func TestProcessDeletionRequest_DeleteCompany_Success(t *testing.T) {
	testPrivacyServer := NewTestPrivacyServer()
	twitchID := "123"

	membership := memberships.Membership{
		CompanyID: common.NewUUID(),
		Role:      "Owner",
		FirstName: "McDudeCool",
	}
	testPrivacyServer.MockMemberships.GetMembershipsByAccountReturns([]memberships.Membership{membership}, nil)
	testPrivacyServer.MockMemberships.GetAdminsByCompanyReturns([]memberships.Membership{}, nil)

	// delete company gets called when user is an owner and no admins exist
	_, err := testPrivacyServer.Server.ProcessDeletionRequest(context.Background(), twitchID)

	require.NoError(t, err)
	require.Equal(t, 1, testPrivacyServer.MockMemberships.GetAdminsByCompanyCallCount())
	require.Equal(t, 0, testPrivacyServer.MockMemberships.MigrateOwnerCallCount()) // nothing to migrate to

	// below are specific to deleting the company
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteGameApplicationsByCompanyCallCount())
	require.Equal(t, 1, testPrivacyServer.MockDeveloperApplications.DeleteDeveloperApplicationByCompanyCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteResourcesByCompanyCallCount())
	require.Equal(t, 1, testPrivacyServer.MockMemberships.DeleteCompanyMembershipsCallCount())
	require.Equal(t, 1, testPrivacyServer.MockBackender.DeleteCompanyCallCount())

}
