package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/common"
	"code.justin.tv/devrel/devsite-rbac/internal/middleware"
)

type logFunc func(ctx context.Context, msg interface{}, fields ...logx.Fields)

func asyncWithLogger(ctx context.Context, f func(context.Context, logFunc) error, logFields ...logx.Fields) {
	bgFields := asyncFields(ctx, logFields...)

	logFunc := func(ctx context.Context, msg interface{}, providedFields ...logx.Fields) {
		// merge twirp and provided fields into ones provided here
		fields := append(bgFields, providedFields...)
		logx.Error(ctx, msg, fields...)
	}

	async(ctx, func(ctx context.Context) error {
		return f(ctx, logFunc)
	}, logFields...)
}

// add twirp and other related fields to provided fields
func asyncFields(ctx context.Context, fields ...logx.Fields) []logx.Fields {
	return append(fields, middleware.LogxFields(ctx))
}

// async does some work in the async of an http request.
// twirpFields are copied out of context.
// any other values to copy out to the new context must be done manually
// ctx cannot be used as the parent because http has a cancel on it already
func async(ctx context.Context, f func(context.Context) error, logFields ...logx.Fields) {
	bCtx, cancel := context.WithTimeout(context.Background(), common.AsyncTimeout)

	bgFields := asyncFields(ctx, logFields...)

	go func(ctx context.Context) {
		defer cancel()
		defer func() {
			if r := recover(); r != nil {
				message := fmt.Sprintf("panic in async job: %s", r)
				logx.Error(ctx, message, bgFields...)
			}
		}()

		if err := f(ctx); err != nil {
			logx.Error(ctx, err, bgFields...)
		}
	}(bCtx)
}
