package rbacrpcserver

import (
	"fmt"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companyinvites"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"context"

	"github.com/twitchtv/twirp"
)

// CreateCompanyInvite creates a new CompanyInvite of a user into an organization
func (s *Server) CreateCompanyInvite(ctx context.Context, params *rbacrpc.CreateCompanyInviteRequest) (*rbacrpc.CompanyInvite, error) {
	if err := s.validateCreateCompanyInviteParams(ctx, params); err != nil {
		return nil, err
	}
	invite := companyinvites.CompanyInvite{
		CompanyID:       params.CompanyId,
		InviteeTwitchID: params.InviteeTwitchId,
		InviterTwitchID: params.InviterTwitchId,
		Role:            params.Role,
	}

	// Ensure company exists
	company, err := s.Backend.SelectCompany(ctx, invite.CompanyID)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("company not found")
	}
	if err != nil {
		return nil, err // internal error
	}

	// Ensure user exists in the users-service (vaidate TwitchID)
	validInvitee, errInvitee := s.Users.IsValidTwitchID(ctx, invite.InviteeTwitchID)
	if errInvitee != nil {
		return nil, errInvitee
	}
	if !validInvitee {
		return nil, twirp.InvalidArgumentError("twitch_id", "is invalid or user with this numeric id does not exist")
	}

	// Ensure user exists in the users-service (vaidate TwitchID)
	validInviter, errInviter := s.Users.IsValidTwitchID(ctx, invite.InviterTwitchID)
	if errInviter != nil {
		return nil, errInviter
	}
	if !validInviter {
		return nil, twirp.InvalidArgumentError("twitch_id", "is invalid or user with this numeric id does not exist")
	}

	// Ensure user doesn't already have invite.
	// Ensure the user is not already a member in the same org (prevent duplicates) or another org (prevent multi-org)
	inviteValidator, inviteValidatorError := s.IsUserValidForCompanyInvite(ctx, &rbacrpc.ValidUserCompanyInviteRequest{
		TwitchId:  params.InviteeTwitchId,
		CompanyId: params.CompanyId,
	})

	if inviteValidatorError != nil {
		return nil, err
	}
	if !inviteValidator.VerifiedEmail {
		return nil, twirp.NewError(twirp.FailedPrecondition, "Invitee's email is not verified")
	}
	if !inviteValidator.Has_2Fa {
		return nil, twirp.NewError(twirp.FailedPrecondition, "Invitee's does not have 2fa enabled")
	}
	if !inviteValidator.NoInvites {
		// if we found an existing invite, lets make sure it isn't expired.
		expiredInvite, unexpiredErr := s.CompanyInvites.ExpiredCompanyInviteForUserOnOrganization(ctx, params.CompanyId, params.InviteeTwitchId)
		if unexpiredErr != nil {
			return nil, err
		}

		if expiredInvite == "" {
			return nil, twirp.NewError(twirp.AlreadyExists, "CompanyInvite already exists")
		} else {
			// if we hit this case that means the only invite we found was an expired one, so lets delete it and allow for them to automatically be reinvited
			clearExpiredErr := s.CompanyInvites.DeleteCompanyInvite(ctx, expiredInvite)
			if clearExpiredErr != nil {
				return nil, err
			}
		}

	}
	if !inviteValidator.NotExistingMember {
		return nil, twirp.NewError(twirp.AlreadyExists, "User already exists in the organization")
	}

	// check on if we are attempting to set a billing manager and if they meet the preconditions
	if params.Role == billingManagerRole {
		if err := s.validateCanBeBillingManager(ctx, params.InviteeTwitchId); err != nil {
			return nil, twirp.NewError(twirp.FailedPrecondition, "User does not meet prerequisites to become a billing manager.")
		}
	}

	// End Checking invitee conditions

	// Ensure permissions
	if !auth.IsWhitelistAdmin(ctx) {
		addUserPermission := permissions.AddUser
		if params.Role == billingManagerRole {
			addUserPermission = permissions.AddUserBillingManager
		}

		// Get requesting user role in company
		requestingMemb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.InviterTwitchId)
		if errorutil.IsErrNoRows(err) {
			return nil, twirp.NewError(twirp.PermissionDenied, "membership for current user (requesting_twitch_id, company_id) not found")
		}
		if err != nil {
			return nil, err
		}

		if !doesRoleHavePermission(requestingMemb.Role, addUserPermission) {
			return nil, twirp.NewError(twirp.PermissionDenied, "cannot invite user to company")
		}

		// If the current users role is Manager, respond with permission denied when the params the role is owner or admin
		if requestingMemb.Role == managerRole {
			if params.Role == adminRole {
				return nil, twirp.NewError(twirp.PermissionDenied, "cannot invite member with role "+params.Role)
			}
		}
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Create CompanyInvite
	inviteResult, errResult := s.CompanyInvites.InsertCompanyInvite(ctx, &invite)
	if errResult != nil {
		return nil, errResult
	}

	if errResult := s.Backend.Commit(ctx); errResult != nil {
		return nil, errResult
	}

	async(ctx, func(ctx context.Context) error {
		// Send email via DART integration here
		emailTraceId, emailError := s.Dart.SendCompanyInviteToUser(ctx, invite.InviteeTwitchID, company.CompanyName) // if email was sent via dart we can track the status of that message with this

		s.auditUserInviteChange(ctx, UserInviteAction{
			CurrentUserTwitchID: invite.InviterTwitchID, // twitchID of the main actor of the update (inviter)
			EntityTwitchID:      invite.InviteeTwitchID, // twitchID of the person being invited (invitee)
			ActionFormat:        fmt.Sprintf("Create: Invite sent to join as role %s with traceid of %s", invite.Role, emailTraceId),
			CompanyID:           invite.CompanyID, // Company/organization the user is being invited to join
		})

		return emailError
	})

	return inviteResult, nil
}

func (s *Server) validateCreateCompanyInviteParams(ctx context.Context, params *rbacrpc.CreateCompanyInviteRequest) error {
	if auth.IsWhitelistAdmin(ctx) {
		params.InviterTwitchId = auth.GetTwitchID(ctx)
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"invitee_twitch_id", params.InviteeTwitchId},
		{"inviter_twitch_id", params.InviterTwitchId},
		{"role", params.Role},
		{"company_id", params.CompanyId},
	}); err != nil {
		return err
	}

	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return err
	}

	// Validate role, and make sure the param is properly formatted to simplify equality checks
	formattedRole, err := SanitizeRole("role", params.Role)
	if err != nil {
		return err
	}
	params.Role = formattedRole

	if params.Role == "Owner" {
		return twirp.NewError(twirp.PermissionDenied, "User can not be set to Owner")
	}

	return nil
}

type UserInviteAction struct {
	CurrentUserTwitchID string // twitchID of the main actor of the update
	EntityTwitchID      string // twitchID of the membership that is being updated
	ActionFormat        string
	CompanyID           string
}

const userInvite string = "UserInvite"

func (s *Server) auditUserInviteChange(ctx context.Context, a UserInviteAction) {
	if a.CurrentUserTwitchID == "" {
		a.CurrentUserTwitchID = auth.GetTwitchID(ctx)
	}

	actionHistory := &actionhistories.ActionHistory{
		UserTwitchID: a.CurrentUserTwitchID,
		Action:       a.ActionFormat,
		EntityType:   userInvite,
		EntityID:     a.EntityTwitchID,
		CompanyID:    common.NewSQLNullString(a.CompanyID),
	}

	s.ActionHistories.InsertActionHistory(ctx, actionHistory)
	logx.Info(ctx, actionHistory.ToLogxFields())

}
