package rbacrpcserver

import (
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/backend/common"

	"context"

	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"

	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *Server) CreateGameApplicationV2(ctx context.Context, params *rbacrpc.CreateGameApplicationRequestV2) (*rbacrpc.GameApplication, error) {
	gameApplicationRequest := &rbacrpc.CreateGameApplicationRequest{
		GameId:    params.GameId,
		CompanyId: params.CompanyId,
	}

	// whitelisted admins can always create game applications
	if !auth.IsWhitelistAdmin(ctx) {
		validResp, err := s.ValidateByTwitchID(ctx, &rbacrpc.ValidateQuery{
			UserId:       params.RequestingTwitchId,
			ResourceId:   params.CompanyId,
			Permission:   permissions.AddGames,
			ResourceType: permissions.Company,
		})
		if err != nil {
			return nil, err
		}
		if !validResp.Valid {
			return nil, twirp.NewError(twirp.PermissionDenied, "cannot add games to this company")
		}
	}

	return s.CreateGameApplication(ctx, gameApplicationRequest)
}

func (s *Server) CreateGameApplication(ctx context.Context, params *rbacrpc.CreateGameApplicationRequest) (*rbacrpc.GameApplication, error) {
	if err := validateCreateGameApplication(ctx, params); err != nil {
		return nil, err
	}

	// If there is already a pending game application by the same company => assume success (idempotency)
	existingGameApp, err := s.Backend.SelectGameApplicationByGameId(ctx, params.GameId)
	if err != nil {
		return nil, err
	}
	if existingGameApp != nil && existingGameApp.CompanyId == params.CompanyId {
		return existingGameApp, nil // success, to be idempotent, assume it was created again.
	}

	// If the game is already owned then return an AlreadyExists error (avoid multiple ownership for now)
	if err := s.verifyResourceNotAlreadyOwned(ctx, "game", fmt.Sprintf("%d", params.GameId)); err != nil {
		return nil, err
	}

	// Create game application
	gameApp, err := s.Backend.InsertGameApplication(ctx, params)
	if err != nil {
		return nil, err
	}
	s.auditCreateGameApplication(ctx, gameApp)
	return gameApp, nil
}

func (s *Server) verifyResourceNotAlreadyOwned(ctx context.Context, resourceType, resourceID string) error {
	// Validate resource isn't already owned
	_, total, err := s.Backend.FindResources(ctx, "", resourceType, resourceID, 1, 0)
	if err != nil {
		return err
	}

	if total > 0 {
		return twirp.NewError(twirp.AlreadyExists, fmt.Sprintf("%s already owned by a company", resourceType))
	}

	return nil
}

func validateCreateGameApplication(ctx context.Context, params *rbacrpc.CreateGameApplicationRequest) error {
	if params.GameId <= 0 {
		return twirp.RequiredArgumentError("game id")
	}
	if params.CompanyId == "" {
		return twirp.RequiredArgumentError("company id")
	}
	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return err
	}
	return nil
}

func (s *Server) auditCreateGameApplication(ctx context.Context, gameApplication *rbacrpc.GameApplication) {
	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: auth.GetTwitchID(ctx),
		Action:       "Game application created",
		EntityType:   "Game",
		EntityID:     fmt.Sprintf("%d", gameApplication.GameId),
		CompanyID:    common.NewSQLNullString(gameApplication.CompanyId),
	})
}
