package rbacrpcserver

import (
	"context"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companyresources"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"github.com/stretchr/testify/require"
)

func TestCreateGameApplication_Success(t *testing.T) {
	server, rbacBackend := NewTestServer()
	rbacBackend.SelectGameApplicationByGameIdReturns(nil, nil)
	rbacBackend.FindResourcesReturns(nil, 0, nil)
	gameApp := rbacrpc.GameApplication{GameId: int32(123), GameName: "MyGame"}
	rbacBackend.InsertGameApplicationReturns(&gameApp, nil)
	rbacBackend.SelectCompanyReturns(&rbacrpc.Company{CompanyName: "mycoName"}, nil)

	resp, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: common.NewUUID(),
	})
	require.NoError(t, err)
	require.Equal(t, gameApp.GameName, resp.GameName)
}

func TestCreateGameApplicationV2_Success(t *testing.T) {
	server, rbacBackend := NewTestServer()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	companyID := common.NewUUID()
	userID := "123"

	rbacBackend.SelectGameApplicationByGameIdReturns(nil, nil)
	rbacBackend.FindResourcesReturns(nil, 0, nil)
	gameApp := rbacrpc.GameApplication{GameId: int32(123), GameName: "MyGame"}
	rbacBackend.InsertGameApplicationReturns(&gameApp, nil)
	rbacBackend.SelectCompanyReturns(&rbacrpc.Company{CompanyName: "mycoName"}, nil)

	fakeMemberships.GetMembershipReturns(memberships.Membership{
		CompanyID: companyID,
		Role:      "Administrator",
		TwitchID:  userID,
	}, nil)

	resp, err := server.CreateGameApplicationV2(ctxBck, &rbacrpc.CreateGameApplicationRequestV2{
		GameId:             int32(123),
		CompanyId:          companyID,
		RequestingTwitchId: userID,
	})
	require.NoError(t, err)
	require.Equal(t, gameApp.GameName, resp.GameName)
}

func TestCreateGameApplicationV2_PermissionDenied(t *testing.T) {
	server, _ := NewTestServer()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	companyID := common.NewUUID()
	userID := "123"

	fakeMemberships.GetMembershipReturns(memberships.Membership{
		CompanyID: companyID,
		Role:      "BadUser",
		TwitchID:  userID,
	}, nil)

	_, err := server.CreateGameApplicationV2(ctxBck, &rbacrpc.CreateGameApplicationRequestV2{
		GameId:             int32(123),
		CompanyId:          companyID,
		RequestingTwitchId: userID,
	})
	require.EqualError(t, err, "twirp error permission_denied: cannot add games to this company")
}

func TestCreateGameApplication_ValidatesCompanyID(t *testing.T) {
	server, _ := NewTestServer()

	_, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: "", // empty!
	})
	require.EqualError(t, err, "twirp error invalid_argument: company id is required")

	_, err = server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: "invalid", // not a valid UUID
	})
	require.EqualError(t, err, "twirp error invalid_argument: company_id invalid UUID format")
}

func TestCreateGameApplication_ExistingApplicationSameCompany(t *testing.T) {
	server, rbacBackend := NewTestServer()

	// Mock same application already existing in the DB
	companyID := common.NewUUID()
	existingGameApp := &rbacrpc.GameApplication{GameId: int32(123), CompanyId: companyID}
	rbacBackend.SelectGameApplicationByGameIdReturns(existingGameApp, nil)

	resp, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: companyID,
	})
	require.NoError(t, err)
	require.Equal(t, existingGameApp.GameId, resp.GameId)
	require.Equal(t, existingGameApp.CompanyId, resp.CompanyId)
}

func TestCreateGameApplication_ExistingApplicationAnotherCompanySuccess(t *testing.T) {
	server, rbacBackend := NewTestServer()

	companyID := common.NewUUID()
	anotherCompanyID := common.NewUUID()

	// Mock same application already existing in the DB, but by another company
	existingGameApp := &rbacrpc.GameApplication{GameId: int32(123), CompanyId: anotherCompanyID}
	rbacBackend.SelectGameApplicationByGameIdReturns(existingGameApp, nil)

	rbacBackend.FindResourcesReturns(nil, 0, nil)
	gameApp := rbacrpc.GameApplication{GameId: int32(123), GameName: "MyGame", CompanyId: companyID}
	rbacBackend.InsertGameApplicationReturns(&gameApp, nil)
	rbacBackend.SelectCompanyReturns(&rbacrpc.Company{CompanyName: "mycoName"}, nil)

	resp, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: companyID,
	})
	require.NoError(t, err)
	require.Equal(t, existingGameApp.GameId, resp.GameId)
	require.NotEqual(t, existingGameApp.CompanyId, resp.CompanyId)
	require.Equal(t, companyID, resp.CompanyId)
}

func TestCreateGameApplication_ExistingGameAlreadyExistsError(t *testing.T) {
	server, rbacBackend := NewTestServer()
	rbacBackend.SelectGameApplicationByGameIdReturns(nil, nil) // no pending application
	rbacBackend.FindResourcesReturns([]companyresources.ResourceType{{}}, 1, nil)

	_, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: common.NewUUID(),
	})
	require.EqualError(t, err, "twirp error already_exists: game already owned by a company")
}

func TestCreateGameApplication_FindApplicationInternalError(t *testing.T) {
	server, rbacBackend := NewTestServer()
	rbacBackend.SelectGameApplicationByGameIdReturns(nil, errUnexpected)

	_, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: common.NewUUID(),
	})
	require.EqualError(t, err, errUnexpected.Error())
}

func TestCreateGameApplication_CreateInternalError(t *testing.T) {
	server, rbacBackend := NewTestServer()
	rbacBackend.SelectGameApplicationByGameIdReturns(nil, nil)
	rbacBackend.FindResourcesReturns(nil, 0, nil)
	rbacBackend.InsertGameApplicationReturns(nil, errUnexpected)

	_, err := server.CreateGameApplication(ctxBck, &rbacrpc.CreateGameApplicationRequest{
		GameId:    int32(123),
		CompanyId: common.NewUUID(),
	})
	require.EqualError(t, err, errUnexpected.Error())
}

func TestValidateCreateGameApplication(t *testing.T) {
	for _, testCase := range []struct {
		name    string
		params  *rbacrpc.CreateGameApplicationRequest
		success bool
	}{
		{
			name: "gameid0",
			params: &rbacrpc.CreateGameApplicationRequest{
				GameId:    0,
				CompanyId: common.NewUUID(),
			},
		},
		{
			name: "gameid-1",
			params: &rbacrpc.CreateGameApplicationRequest{
				GameId:    -1,
				CompanyId: common.NewUUID(),
			},
		},
		{
			name: "no-company",
			params: &rbacrpc.CreateGameApplicationRequest{
				GameId: 1,
			},
		},
		{
			name: "valid",
			params: &rbacrpc.CreateGameApplicationRequest{
				GameId:    1,
				CompanyId: common.NewUUID(),
			},
			success: true,
		},
	} {
		t.Run(testCase.name, func(t *testing.T) {
			err := validateCreateGameApplication(context.Background(), testCase.params)
			if testCase.success {
				require.NoError(t, err)
			} else {
				require.Error(t, err)
			}
		})
	}
}
