package rbacrpcserver

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	jwt "github.com/dgrijalva/jwt-go"
	"github.com/twitchtv/twirp"
)

// A simple converter from VHS Operations to RBAC Permissions.
// This helps us construct the verification request towards RBAC to see if the user
// has the sufficient permissions to execute the requested operation
var (
	vhsPermission = map[string]string{
		"get_campaigns_by_account": permissions.ViewDrops,
		"get_campaign_by_id":       permissions.ViewDrops,
		"post_campaign":            permissions.CreateDrops,
		"put_campaign":             permissions.ManageDrops,
		"get_service":              permissions.ViewDrops,
		"put_service":              permissions.ManageDrops,
		"post_campaign_image_url":  permissions.CreateDrops,
		"post_item_image_url":      permissions.CreateDrops,
		"get_all_items":            permissions.ViewDrops,
		"get_item":                 permissions.ViewDrops,
		"create_item":              permissions.CreateDrops,
		"update_item":              permissions.ManageDrops,
	}
)

// CreateJWT Create a JWT To be used for Drops
func (s *Server) CreateJWT(ctx context.Context, request *rbacrpc.JWTRequest) (*rbacrpc.JWT, error) {
	// visage currently handles getting company for user and using getAccountID
	accountID := request.CompanyId

	if request.UserId != "" { // TODO: remove when visage updates to include this logic below.
		var company *rbacrpc.Company
		if request.CompanyId == "" { // TODO: Update twilight to allows provide this
			membershipsResp, err := s.ListCompanyMemberships(ctx, &rbacrpc.ListCompanyMembershipsRequest{
				TwitchId: request.UserId,
			})

			if err != nil {
				return nil, err
			}

			if len(membershipsResp.Memberships) == 0 {
				return nil, twirp.NewError(twirp.PermissionDenied, "not in company")
			}

			// take first membership company
			company = membershipsResp.Memberships[0].Company
			request.CompanyId = company.Id
		}

		permissionName, ok := vhsPermission[request.Operation]
		if !ok {
			return nil, twirp.InvalidArgumentError("operation", "Invalid Operation")
		}

		valid, err := s.ValidateByTwitchID(ctx, &rbacrpc.ValidateQuery{
			UserId:       request.UserId,
			Permission:   permissionName,
			ResourceId:   request.CompanyId,
			ResourceType: "company",
		})
		if err != nil {
			return nil, err
		}
		if !valid.Valid {
			return nil, twirp.NewError(twirp.PermissionDenied, "cannot create jwt")
		}

		if company == nil {
			company, err = s.GetCompany(ctx, &rbacrpc.Id{
				Id: request.CompanyId,
			})
			if err != nil {
				return nil, err
			}
		}

		accountID = getAccountID(company)
	}

	exp := time.Now()
	exp = exp.Add(time.Minute * time.Duration(60))

	token := jwt.NewWithClaims(jwt.SigningMethodHS512, jwt.MapClaims{
		"operation":  request.Operation,
		"account_id": accountID,
		"exp":        exp.Unix(),
	})

	// need to add a breakline for backwards compatibility (https://git-aws.internal.justin.tv/devrel/devsite-rbac/pull/31/files#diff-3baf47c64847a8fb8aaa8cc2e088513bR116)
	dropsKey := s.DropsJWTKey + `
`

	tokenString, err := token.SignedString([]byte(dropsKey))
	if err != nil {
		return nil, err
	}

	userToken := &rbacrpc.JWT{Token: tokenString, AccountId: accountID}
	return userToken, nil
}

// getAccountID gets the drop campaign account from a company.
// drops have been around through the multiple devsite rbac iterations
// and as such have used various identifiers from the iterations.
func getAccountID(c *rbacrpc.Company) string {
	var companyID string
	if c.Legacy { // legacy company
		companyID = c.Id
	} else if c.CurseCompanyId != -1 { // curse company
		companyID = fmt.Sprint(c.CurseCompanyId)
	} else { // company this codebase created
		companyID = c.Id
	}
	return companyID
}
